
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.erp.service.impl;

import com.baijia.tianxiao.biz.erp.dto.OrgCourseInfoPcDto;
import com.baijia.tianxiao.biz.erp.dto.response.exportCourse.CourseStudent;
import com.baijia.tianxiao.biz.erp.dto.response.exportCourse.ExportCourseInfoDto;
import com.baijia.tianxiao.biz.erp.dto.response.schedule.CommonInfoTipsDto;
import com.baijia.tianxiao.biz.erp.dto.response.schedule.CourseBaseInfoDto;
import com.baijia.tianxiao.biz.erp.dto.response.schedule.CourseNameAndIdDto;
import com.baijia.tianxiao.biz.erp.dto.response.schedule.VipClassDetailDto;
import com.baijia.tianxiao.biz.erp.service.CourseLessonService;
import com.baijia.tianxiao.biz.erp.service.ErpCourseService;
import com.baijia.tianxiao.biz.erp.service.ErpStudentService;
import com.baijia.tianxiao.biz.erp.teacherCenter.service.ExcelCourseExportService;
import com.baijia.tianxiao.constant.AvatarConstants;
import com.baijia.tianxiao.dal.constant.ChargeUnit;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseConsumeRuleDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonSignDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.dao.TXCascadeAccountDao;
import com.baijia.tianxiao.dal.org.dao.TXCascadeCredentialDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.OrgCourseConsumeRule;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.org.po.OrgStudentCourse;
import com.baijia.tianxiao.dal.org.po.TXCascadeAccount;
import com.baijia.tianxiao.dal.org.po.TXCascadeCredential;
import com.baijia.tianxiao.dto.IdNameDto;
import com.baijia.tianxiao.dto.PersonBaseDto;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.excel.dto.ExportField;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoMContext;
import com.baijia.tianxiao.filter.TianxiaoPCContext;
import com.baijia.tianxiao.sal.course.dto.OrgCourseInfoDto;
import com.baijia.tianxiao.sal.course.dto.response.TeacherResponseDto;
import com.baijia.tianxiao.sal.course.enums.CourseConsumeRuleEnum;
import com.baijia.tianxiao.sal.course.service.CourseStudentService;
import com.baijia.tianxiao.sal.course.service.CourseTeacherService;
import com.baijia.tianxiao.sal.course.service.OrgCourseConsumeRuleService;
import com.baijia.tianxiao.sal.course.service.OrgCourseService;
import com.baijia.tianxiao.sal.organization.constant.CascadeType;
import com.baijia.tianxiao.sal.organization.constant.DeviceType;
import com.baijia.tianxiao.sal.organization.constant.TXPermissionConst;
import com.baijia.tianxiao.sal.organization.org.service.TxAccountPermissionService;
import com.baijia.tianxiao.sal.room.dto.ClassRoomDto;
import com.baijia.tianxiao.sal.room.service.ClassRoomService;
import com.baijia.tianxiao.sal.student.enums.StudentErrorCode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.BaseUtils;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Transformer;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.*;

/**
 * @author zhangbing
 * @version 1.0
 * @title ErpCourseServiceImpl
 * @desc TODO
 * @date 2016年8月25日
 */
@Service
@Slf4j
public class ErpCourseServiceImpl implements ErpCourseService {

    // 默认班主任为：未设置
    private static final String DEFAULT_CLASS_MASTER = "未设置";
    private static final List<String> exportHeader = Lists.newArrayList("班级名称", "授课老师", "课程类型", "教室", "学员人数", "班级学员", "学员电话", "排课课次", "已上课次", "剩余课次");

    @Resource
    private OrgCourseDao orgCourseDao;
    @Resource
    private CourseTeacherService courseTeacherService;
    @Resource
    private TXCascadeAccountDao cascadeAccountDao;
    @Resource
    private ClassRoomService classRoomService;
    @Resource
    private TXCascadeCredentialDao credentialDao;
    @Resource
    private OrgStudentCourseDao orgStudentCourseDao;
    @Resource
    private OrgClassLessonDao orgClassLessonDao;
    @Resource
    private OrgCourseService orgCourseService;
    @Resource
    private ErpStudentService erpStudentService;
    @Resource
    private OrgCourseConsumeRuleDao orgCourseConsumeRuleDao;
    @Resource
    private OrgAccountDao orgAccountDao;
    @Resource
    private CourseStudentService courseStudentService;
    @Resource
    private OrgInfoDao orgInfoDao;
    @Resource
    private TxAccountPermissionService permissionService;
    @Resource
    private OrgLessonSignDao orgLessonSignDao;
    @Resource
    private OrgCourseConsumeRuleService orgCourseConsumeRuleService;
    @Resource
    private OrgStudentDao orgStudentDao;
    @Resource
    private OrgStudentLessonDao orgStudentLessonDao;
    @Resource
    private CourseLessonService courseLessonService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CourseBaseInfoDto getCourseDetailById(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId");
        Preconditions.checkArgument(courseId != null && courseId > 0, "illegal courseId");
        CourseBaseInfoDto orgCourseInfoDto = new CourseBaseInfoDto();

        OrgCourse course = null;
        if (courseId != null && courseId > 0) {
            course = this.orgCourseDao.getById(courseId);
            if (course != null) {
                orgCourseInfoDto.setCourseNumber(course.getNumber());
                orgCourseInfoDto.setCourseId(courseId);
                orgCourseInfoDto.setCourseName(course.getName());
                orgCourseInfoDto.setCoursePrice(course.getPrice());
                orgCourseInfoDto.setAddress(course.getAddress());
                orgCourseInfoDto.setStartTime(course.getBeginTime());
                orgCourseInfoDto.setEndTime(course.getEndTime());
                orgCourseInfoDto.setCourseType(course.getCourseType());

                // 总课次
                orgCourseInfoDto.setTotalLessons(course.getFreq() == null ? 0 : course.getFreq());
                orgCourseInfoDto.setTeacherInfo(this.getCourseTeacherTips(orgId, courseId));
                orgCourseInfoDto.setClassMaster(getAccountName(orgId, course.getCascadeId()));
                orgCourseInfoDto.setRoomInfo(this.getCourseRoomTips(orgId, courseId));
                if (course.getChargeUnit() == ChargeUnit.BY_HOUR.getCode()) {
                    orgCourseInfoDto.setFinishedLessons(this.getFinishedLessonMinute(orgId, courseId));
                } else {
                    orgCourseInfoDto.setFinishedLessons(this.getFinishedLessonCount(orgId, courseId));
                }
                orgCourseInfoDto.setStudentCount(this.getCourseStudentCount(orgId, courseId));
                // 按期班课/按次班课
                orgCourseInfoDto.setChargeType(course.getChargeType());
                orgCourseInfoDto.setChargeUnit(course.getChargeUnit());

                // 获取课消规则信息
                String consumRuleName = "未设置课消规则";
                OrgCourseConsumeRule courseConsumRule = orgCourseConsumeRuleDao.getRuleByCourseId(orgId, courseId);
                if (courseConsumRule != null) {
                    orgCourseInfoDto.setConsumRuleId(courseConsumRule.getId());
                    int consumVal = courseConsumRule.getRuleValue();
                    consumRuleName = CourseConsumeRuleEnum.getRuleDescByValue(consumVal);
                }
                orgCourseInfoDto.setConsumeRuleName(consumRuleName);
                orgCourseInfoDto.setMaxStudent(course.getMaxStudent());
            }
        }
        log.info("ErpCourseServiceImpl.getCourseDetailById.orgId:{}, courseId:{}, data:{}", orgId, courseId,
                orgCourseInfoDto);
        return orgCourseInfoDto;
    }

    @Override
    public VipClassDetailDto getVipClassDetailById(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId");
        Preconditions.checkArgument(courseId != null && courseId > 0, "illegal courseId");
        VipClassDetailDto vipClassDetailDto = new VipClassDetailDto();
        OrgCourse classInfo = orgCourseDao.getById(courseId);// 班级信息
        if (classInfo == null || classInfo.getIsClass() != CourseTypeEnum.IS_CLASS_TRUE.getCode()
                || classInfo.getIsDel() == DeleteStatus.DELETED.getValue()) {
            throw new BussinessException(CommonErrorCode.NOT_FOUND, "班级不存在或已被删除");
        }
        OrgCourse orgCourseInfo = orgCourseDao.getById(classInfo.getParentId());// 课程信息
        // if(orgCourseInfo == null || orgCourseInfo.getIsCourse() != CourseTypeEnum.IS_COURSE_TRUE.getCode() ||
        // orgCourseInfo.getIsDel() == DeleteStatus.DELETED.getValue()) {
        // throw new BussinessException(CommonErrorCode.NOT_FOUND, "班级所属的课程不存在或已被删除");
        // }
        vipClassDetailDto.setCourseId(classInfo.getId());// 班级id
        vipClassDetailDto.setClassName(classInfo.getName());// 班级名称
        vipClassDetailDto.setTeacherInfo(getCourseTeacherTips(orgId, classInfo.getId()));// 老师信息
        vipClassDetailDto.setRoomInfo(getCourseRoomTips(orgId, classInfo.getId()));// 教室信息
        vipClassDetailDto.setClassMaster(getAccountName(orgId, classInfo.getCascadeId()));// 班主任信息
        if (classInfo.getChargeUnit() == ChargeUnit.BY_HOUR.getCode()) {
            vipClassDetailDto.setFinishedLessons(this.getFinishedVipLessonLength(orgId, courseId));// 已上课时
            vipClassDetailDto.setTotalLessons(this.getFinishedVipBuyLength(orgId, courseId));
        } else {
            vipClassDetailDto.setFinishedLessons(this.getFinishedVipLessonCount(orgId, courseId, orgCourseInfo.getId()));// 已上课次
            vipClassDetailDto.setTotalLessons(this.getFinishedVipBuyLength(orgId, courseId));// 总课节数（课次）
        }
        vipClassDetailDto.setCoursePrice(classInfo.getPrice() == null ? 0 : classInfo.getPrice());// 课程单价
        vipClassDetailDto.setCreateTime(classInfo.getCreateTime());// 创建时间
        vipClassDetailDto.setStudentCount(getCourseStudentCount(orgId, courseId));// 班级人数
        vipClassDetailDto.setStudentStatus(vipClassDetailDto.getStudentCount() == 0 ? 1 : 0);// 班级状态 班级人数为0表示已退班
        // Double totalPrice = orgStudentCourseDao.getVipClassTotalPrice(orgId, courseId);
        // vipClassDetailDto.setTotalPrice(totalPrice == null ? 0 : totalPrice);//课程总价
        vipClassDetailDto.setTotalPrice(0.0);// 课程总价
        vipClassDetailDto.setAddress(classInfo.getAddress());
        // 课消规则 备注：不按班级设置课消，按课程设置课消
        OrgCourseConsumeRule courseConsumRule = orgCourseConsumeRuleDao.getRuleByCourseId(orgId, orgCourseInfo.getId());
        if (courseConsumRule != null) {
            int consumeVal = courseConsumRule.getRuleValue();
            vipClassDetailDto.setConsumeRuleId(consumeVal);// 课消规则编码
            vipClassDetailDto.setConsumeRuleName(CourseConsumeRuleEnum.getRuleDescByValue(consumeVal));// 课消规则描述语
        } else {
            throw new BussinessException(CommonErrorCode.NOT_FOUND, "课消规则未设置");
        }
        vipClassDetailDto.setChargeType(classInfo.getChargeType());
        vipClassDetailDto.setChargeUnit(classInfo.getChargeUnit());
        return vipClassDetailDto;
    }

    private Integer getFinishedVipBuyLength(Long orgId, Long courseId) {
        List<OrgStudentCourse> list = orgStudentCourseDao.getStudentListByCourseId(orgId, courseId);
        if (CollectionUtils.isEmpty(list)) {
            return 0;
        }
        OrgStudentCourse studentCourse = list.get(0);
        return studentCourse.getLessonCount();
    }

    private Integer getFinishedVipLessonLength(Long orgId, Long courseId) {
        Map<Long, Map<Long, Integer>> finishLessonLengthMap = orgClassLessonDao.getFinishLessonLengthMap(orgId, Arrays.asList(courseId), DeleteStatus.NORMAL.getValue());
        Map<Long, Integer> fMap = finishLessonLengthMap.get(courseId);
        int finish = fMap != null ? intSum(fMap.values()) : 0;
        return finish;
    }

    /**
     * @param orgId          ： 机构ID
     * @param courseId       ： 班级id
     * @param parentCourseId ： 班级对应的课程的课程id
     * @return
     */
    private Integer getFinishedVipLessonCount(Long orgId, Long courseId, Long parentCourseId) {
        OrgCourseConsumeRule orgCourseConsumeRule = orgCourseConsumeRuleDao.getRuleByCourseId(orgId, parentCourseId);
        int ruleValue = orgCourseConsumeRule.getRuleValue();
        int count = orgLessonSignDao.getVipLessonSignCount(orgId, courseId, ruleValue);
        return count;
    }

    /**
     * 根据cdb.org_course.cascade_id获取班级班主任名称
     *
     * @param orgId
     * @param cascadeId
     * @return
     */
    private String getAccountName(Long orgId, Integer cascadeId) {
        if (cascadeId != null && cascadeId > 0) {
            TXCascadeAccount cascadeAccount = cascadeAccountDao.getById(cascadeId);
            if (cascadeAccount == null || cascadeAccount.getIsdel() == 1 || cascadeAccount.getStatus() == 1) {
                log.warn("[Wechat] Message TXCascadeAccount is invalidate.cascadeAccount={}", cascadeAccount);
            } else {
                TXCascadeCredential credential =
                        credentialDao.getTXCascadeCredentialByCredentialId(cascadeAccount.getCredentialId());
                if (credential != null) {
                    return credential.getName() == null ? credential.getMobile() : credential.getName();
                } else {
                    log.error("[Wechat] Message TXCascadeCredential is invalidate.credentialId="
                            + cascadeAccount.getCredentialId());
                }
            }
        } else if (cascadeId == 0) {
            OrgInfo orgInfo = orgInfoDao.getOrgInfo(orgId.intValue(), "contacts");
            return (orgInfo != null && StringUtils.isNotBlank(orgInfo.getContacts())) ? orgInfo.getContacts() : "";
        }
        return DEFAULT_CLASS_MASTER;
    }

    /**
     * 获取课程教室相关信息
     *
     * @param orgId
     * @param courseId
     * @return
     */
    private CommonInfoTipsDto getCourseRoomTips(Long orgId, Long courseId) {
        List<ClassRoomDto> data = this.classRoomService.listOrgCourseRooms(orgId, courseId);
        CommonInfoTipsDto tips = null;
        if (CollectionUtils.isNotEmpty(data)) {
            tips = new CommonInfoTipsDto();
            int count = data.size();
            Set<String> names = BaseUtils.getPropertiesList(data, "roomName");
            tips.setCount(count);
            tips.setNames(names);
        }
        log.info("getCourseRoomTips.tips:{}", tips);
        return tips;
    }

    /**
     * 获取课程老师相关信息
     *
     * @param orgId
     * @param courseId
     * @return
     */
    private CommonInfoTipsDto getCourseTeacherTips(Long orgId, Long courseId) {
        List<TeacherResponseDto> data = this.courseTeacherService.listCourseTeacher(orgId, courseId);
        CommonInfoTipsDto tips = null;
        if (CollectionUtils.isNotEmpty(data)) {
            tips = new CommonInfoTipsDto();
            int count = data.size();
            Set<String> names = BaseUtils.getPropertiesList(data, "teacherName");
            tips.setCount(count);
            tips.setNames(names);
        }
        log.info("getCourseTeacherTips.tips:{}", tips);
        return tips;
    }

    /**
     * 获取班级学生数
     *
     * @param orgId
     * @param courseId
     * @return
     */
    private int getCourseStudentCount(Long orgId, Long courseId) {
        // 学生数量
        List<Long> studentIds = this.orgStudentCourseDao.getStudents(orgId, courseId, 0);
        return CollectionUtils.isNotEmpty(studentIds) ? studentIds.size() : 0;
    }

    /**
     * 查询已上完的课时数
     *
     * @param orgId
     * @param courseId
     * @return
     */
    private int getFinishedLessonMinute(Long orgId, Long courseId) {
        int finishedLessonCount = 0;
        Map<Long, Map<Long, Integer>> finishLessonLengthMap = orgClassLessonDao.getFinishLessonLengthMap(orgId, Arrays.asList(courseId), DeleteStatus.NORMAL.getValue());
        Map<Long, Integer> lengthMap = finishLessonLengthMap.get(courseId);
        finishedLessonCount = lengthMap != null ? intSum(lengthMap.values()) : 0;
        return finishedLessonCount;
    }

    /**
     * 查询已上完的课节数
     *
     * @param orgId
     * @param courseId
     * @return
     */
    private int getFinishedLessonCount(Long orgId, Long courseId) {
        int finishedLessonCount =
                this.orgClassLessonDao.getFinishLessonCount(orgId, courseId, DeleteStatus.NORMAL.getValue());
        return finishedLessonCount;
    }

    private Integer intSum(Collection<Integer> values) {
        int sum = 0;
        if (CollectionUtils.isNotEmpty(values)) {
            for (Integer value : values) {
                sum += value;
            }
        }
        return sum;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrUpdateCourse(Long orgId, OrgCourseInfoPcDto course) {
        Long courseId = this.orgCourseService.saveOrUpdateCourse(orgId, course, true, true);
        if (courseId == null || courseId <= 0) {
            throw new BussinessException(CommonErrorCode.SYSTEM_ERROR, "保存课程失败");
        }

        List<Long> roomIds = Lists.newArrayList();
        List<Long> studentIds = Lists.newArrayList();
        try {
            if (StringUtils.isNoneBlank(course.getRoomIds())) {
                roomIds = JacksonUtil.str2List(course.getRoomIds(), Long.class);
            }
            if (StringUtils.isNoneBlank(course.getStudentIds())) {
                studentIds = JacksonUtil.str2List(course.getStudentIds(), Long.class);
            }
        } catch (Exception e) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "教室、学生参数错误");
        }

        // 保存教室
        this.classRoomService.saveCourseRooms(orgId, courseId, roomIds);

        // 保存学生
        this.erpStudentService.saveCourseStudents(orgId, courseId, studentIds);

        // 保存(分配)班主任,参数cascadeId:-1清空班主任；0主账号；null无分配班主任权限，班主任不变；>0子账号
        if (course.getCascadeId() != null) {
            Integer curCascadeId = TianxiaoPCContext.getTXCascadeId();
            // 子账号，需要判断是否有分配班主任的权限
            if (curCascadeId != null && curCascadeId > 0) {
                boolean hasPermission = this.permissionService.checkPermission(orgId, curCascadeId, DeviceType.PC,
                        TXPermissionConst.DISTRIBUTE_CLASSMANAGER.getpCode());
                if (!hasPermission) {
                    throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "没有分配班主任的权限");
                }
            }

            OrgCourse orgCourse = this.orgCourseDao.getByCourseId(courseId, "id", "cascadeId");
            if (course.getCascadeId() == 0) {
                orgCourse.setCascadeId(0);
            } else if (course.getCascadeId() < 0) {
                orgCourse.setCascadeId(-1);
            } else {
                TXCascadeAccount cascadeAccount =
                        this.cascadeAccountDao.getByIdAndOrgId(course.getCascadeId().intValue(), orgId.intValue());
                if (cascadeAccount != null && cascadeAccount.getCredentialId() != null
                        && cascadeAccount.getCredentialId() > 0) {
                    TXCascadeCredential txCascadeCredential =
                            this.credentialDao.getTXCascadeCredentialByCredentialId(cascadeAccount.getCredentialId());
                    if (txCascadeCredential == null) {
                        throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号异常");
                    } else {
                        orgCourse.setCascadeId(cascadeAccount.getId());
                    }
                } else {
                    throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号异常");
                }
            }
            this.orgCourseDao.update(orgCourse, false, "cascadeId");
        }
        return courseId;
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<CourseNameAndIdDto> fuzzyQuery(Long orgId, String courseName, Integer type, PageDto pageDto) {
        Set<Long> courseIds = new HashSet<>();

        // 如果是员工账号,只能查询自己是班主任的课程
        // 如果是校长、主管并且主管拥有排课权限，查询所有课程
        if (TianxiaoPCContext.getTXCascadeId() != null) {
            TXCascadeAccount txCascadeAccount = cascadeAccountDao.getById(TianxiaoPCContext.getTXCascadeId());
            if (txCascadeAccount == null) {
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号不存在");
            }
            if (txCascadeAccount.getAccountType() == CascadeType.STAFF.getValue()) {
                courseLessonService.handleStaffCourseIds(orgId,TianxiaoPCContext.getTXCascadeId(),courseIds);
                if(courseIds.isEmpty()){
                    return Lists.newArrayList();
                }
            }
        }

        OrgAccount orgAccount = orgAccountDao.getById(TianxiaoPCContext.getOrgId());
        List<OrgCourse> courses =
                this.orgCourseDao.fuzzyQueryByCourseType(orgAccount.getNumber(), courseName, courseIds, type, pageDto);

        List<CourseNameAndIdDto> data = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(courses)) {
            data = (List<CourseNameAndIdDto>) CollectionUtils.collect(courses, new Transformer() {

                @Override
                public Object transform(Object input) {
                    CourseNameAndIdDto dto = new CourseNameAndIdDto();
                    OrgCourse course = (OrgCourse) input;
                    dto.setId(course.getId());
                    dto.setName(course.getName());
                    dto.setCourseType(course.getCourseType());
                    return dto;
                }
            });
        }
        return data;
    }

    @Override
    public OrgCourseInfoPcDto getOrgCourseInfoPc(Long orgId, Long courseId)
            throws IllegalAccessException, InvocationTargetException {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        OrgAccount orgAccount = this.orgAccountDao.getById(orgId, "isDel", "mobile");
        OrgInfo orgInfo = this.orgInfoDao.getOrgInfo(orgId.intValue(), "contacts");
        if (orgAccount == null || orgAccount.getIsDel() == DeleteStatus.DELETED.getValue() || orgInfo == null) {
            throw new BussinessException(StudentErrorCode.ORG_NOT_EXIST);
        }

        OrgCourse orgCourse = this.orgCourseDao.getByCourseId(courseId, "id", "cascadeId", "chargeUnit");
        if (orgCourse == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构id错误");
        }

        OrgCourseInfoPcDto courseDto = new OrgCourseInfoPcDto();
        OrgCourseInfoDto courseInfo = this.orgCourseService.getOrgCourseInfo(orgId, courseId);
        if (courseInfo == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "参数错误");
        } else {
            BeanUtils.copyProperties(courseDto, courseInfo);

            // 教室信息重新封装，用于前端统一处理
            List<ClassRoomDto> rooms = this.classRoomService.listOrgCourseRooms(orgId, courseId);
            List<IdNameDto> roomList = Lists.newArrayList();
            for (ClassRoomDto room : rooms) {
                IdNameDto roomDto = new IdNameDto();
                roomDto.setId(room.getRoomId().intValue());
                roomDto.setName(room.getRoomName());
                roomList.add(roomDto);
            }
            courseDto.setRooms(roomList);
            courseDto.setTotalLessons(courseInfo.getFreq() == null ? 0 : courseInfo.getFreq());// 课节数量
            Integer lessonCount = orgClassLessonDao.getLessonCount(orgId, courseId, 0);
            Map<Long, Map<Long, Integer>> lessonLength = orgClassLessonDao.getLessonLengthMap(orgId, Arrays.asList(courseId), DeleteStatus.NORMAL.getValue());
            Map<Long, Integer> map = lessonLength.get(courseId);
            if (orgCourse.getChargeUnit() == ChargeUnit.BY_HOUR.getCode()) {
                courseDto.setScheduleLessons(map != null ? intSum(map.values()) : 0);// 已排课时
            } else {
                courseDto.setScheduleLessons(lessonCount == null ? 0 : lessonCount);// 已排课节
            }

            // 学生
            List<PersonBaseDto> students = this.courseStudentService.getCourseStudentsBase(orgId, courseId);
            courseDto.setStudents(students);
            // 老师信息重新封装，用于前端统一处理
            List<TeacherResponseDto> teachers = courseInfo.getTeachers();
            List<PersonBaseDto> teacherList = Lists.newArrayList();
            if (CollectionUtils.isNotEmpty(teachers)) {
                for (TeacherResponseDto teacher : teachers) {
                    PersonBaseDto teacherDto = new PersonBaseDto();
                    teacherDto.setId(teacher.getTeacherId());
                    teacherDto.setName(teacher.getTeacherName());
                    teacherDto.setAvatarUrl(teacher.getAvatar());
                    teacherList.add(teacherDto);
                }
            }
            courseDto.setCourseTeachers(teacherList);

            // 班主任
            if (orgCourse.getCascadeId() == null || orgCourse.getCascadeId() == 0) {
                // 主帐号
                courseDto.setCascadeId(0l);
                courseDto.setCascadeName(orgInfo.getContacts());
            } else if (orgCourse.getCascadeId() == -1) {
                courseDto.setCascadeId(-1l);
            } else {
                TXCascadeAccount cascadeAccount =
                        this.cascadeAccountDao.getByIdAndOrgId(orgCourse.getCascadeId(), orgId.intValue());
                // 子账号为null的处理
                if (cascadeAccount == null) {
                    courseDto.setCascadeId(0l);
                    courseDto.setCascadeName(orgInfo.getContacts());
                } else if (cascadeAccount.getCredentialId() != null && cascadeAccount.getCredentialId() > 0) {
                    TXCascadeCredential cascadeCredential =
                            this.credentialDao.getTXCascadeCredentialByCredentialId(cascadeAccount.getCredentialId());
                    courseDto.setCascadeId(cascadeAccount.getId().longValue());
                    if (cascadeCredential != null) {
                        courseDto.setCascadeName(cascadeCredential.getName());
                        if (StringUtils.isNoneBlank(cascadeCredential.getAvatar())) {
                            courseDto.setCascadeAvatarUrl(cascadeCredential.getAvatar());
                        } else {
                            courseDto.setCascadeAvatarUrl(AvatarConstants.CONSULT_AVATAR_URL);
                        }
                    }
                }
            }
        }
        return courseDto;
    }

    @Override
    public void exportCourseInfo(HttpServletRequest request, HttpServletResponse response, Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId");
        Preconditions.checkArgument(courseId != null && courseId > 0, "illegal courseId");
        OrgCourse course = this.orgCourseDao.getById(courseId);
        if (course == null || course.getIsDel() == DeleteStatus.DELETED.getValue()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课程不存在或已被删除!");
        }
        ExportCourseInfoDto data = this.getExportCourseInfo(orgId, course);
        this.export(request, response, data);
    }

    /**
     * 获取待导出班级信息
     *
     * @param orgId
     * @param course
     * @return
     */
    private ExportCourseInfoDto getExportCourseInfo(Long orgId, OrgCourse course) {
        ExportCourseInfoDto data = new ExportCourseInfoDto();
        data.setCourseId(course.getId());
        data.setCourseName(course.getName());
        data.setCourseType(course.getCourseType() == CourseTypeEnum.COURSE_TYPE_CLASS.getCode() ? CourseTypeEnum.COURSE_TYPE_CLASS.getMsg() : CourseTypeEnum.COURSE_TYPE_1v1.getMsg());

        CommonInfoTipsDto teacherDto = this.getCourseTeacherTips(orgId, course.getId());
        if (teacherDto != null && !CollectionUtils.isEmpty(teacherDto.getNames())) {
            data.setTeachers(StringUtils.join(teacherDto.getNames(), ","));
        }
        CommonInfoTipsDto classRoomDto = this.getCourseRoomTips(orgId, course.getId());
        if (classRoomDto != null && !CollectionUtils.isEmpty(classRoomDto.getNames())) {
            data.setClassRooms(StringUtils.join(classRoomDto.getNames(), ","));
        }

        List<CourseStudent> courseStudents = this.getCourseStudents(orgId, course.getId());
        data.setStudentList(courseStudents);
        data.setStudentCount(CollectionUtils.isEmpty(courseStudents) ? 0 : courseStudents.size());
        return data;
    }

    /**
     * 获取待导出班级学员信息
     *
     * @param orgId
     * @param courseId
     * @return
     */
    private List<CourseStudent> getCourseStudents(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");

        List<CourseStudent> result = new ArrayList<>();

        List<OrgStudentCourse> courseStudents = this.orgStudentCourseDao.getStudentListByCourseId(orgId, courseId);
        if (CollectionUtils.isEmpty(courseStudents)) {
            return result;
        }

        List<Long> normalStudents = com.google.common.collect.Lists.newArrayList();
        for (OrgStudentCourse orgStudentCourse : courseStudents) {
            if (orgStudentCourse.getStatus().intValue() == 0) {
                normalStudents.add(orgStudentCourse.getUserId());
            }
        }
        if (CollectionUtils.isEmpty(normalStudents)) {
            return result;
        }

        List<OrgStudent> orgStudents = this.orgStudentDao.getStudents(orgId, normalStudents, "", DeleteStatus.NORMAL.getValue(), null, "id", "name", "userId", "mobile");
        if (CollectionUtils.isNotEmpty(orgStudents)) {
            Set<Long> queryStudents = BaseUtils.getPropertiesList(orgStudents, "userId");

            // 学生班级总课节
            Map<Long, List<Long>> studentLessonMap = this.orgStudentLessonDao.getStudentCourseLessonMap(orgId, courseId, queryStudents);
            // 剩余课节
            Map<Long, Integer> studentLeftLessonMap = Maps.newHashMap();
            Integer ruleValue = this.orgCourseConsumeRuleService.getRuleValueByCourseId(orgId, courseId);
            if (ruleValue == null || ruleValue == 0) {
                // 默认按时间节点计算课消
                studentLeftLessonMap = this.orgStudentLessonDao.getStudentsLeftLessonMap(orgId, courseId, queryStudents);
            } else {
                // 各种签到状态，用于按到课/请假/缺课算课消时使用
                Map<Long, List<Integer>> studentSignStatusMap = this.orgLessonSignDao.getOrgCourseStudentSignMap(queryStudents, courseId);
                for (Long userId : queryStudents) {
                    int finish = 0;
                    if (studentSignStatusMap.containsKey(userId)) {
                        finish = CourseConsumeRuleEnum.calculateConsumeTimesBySignCodeList(ruleValue, studentSignStatusMap.get(userId));
                    }
                    // 有的学生无排课信息,需要判断
                    if (studentLessonMap.containsKey(userId)) {
                        studentLeftLessonMap.put(userId, studentLessonMap.get(userId).size() - finish);
                    }
                }
            }

            for (OrgStudent orgStudent : orgStudents) {
                CourseStudent dto = new CourseStudent();
                dto.setStudentId(orgStudent.getId());
                dto.setStudentName(orgStudent.getName());
                dto.setMobile(orgStudent.getMobile());
                dto.setTotalLessons(studentLessonMap.containsKey(orgStudent.getUserId()) ? studentLessonMap.get(orgStudent.getUserId()).size() : 0);
                dto.setLeftLessons(studentLeftLessonMap.containsKey(orgStudent.getUserId()) ? studentLeftLessonMap.get(orgStudent.getUserId()) : 0);
                dto.setFinishLessons(dto.getTotalLessons() - dto.getLeftLessons());
                result.add(dto);
            }
        }
        return result;
    }

    /**
     * 导出班级信息(生成excel)
     *
     * @param response
     * @param data
     */
    private void export(HttpServletRequest request, HttpServletResponse response, ExportCourseInfoDto data) {
        String fileName = data.getCourseName() + "-班级导出信息.xlsx";
        SXSSFWorkbook workbook = null;
        try {
            workbook = new SXSSFWorkbook(200);
            HashMap cellStyleMap = new HashMap();

            List<ExportField> exportFields = Lists.newArrayList();
            for (String header : exportHeader) {
                exportFields.add(new ExportField(header, "@", 5120));
            }
            ExcelCourseExportService.createCourseSheet(workbook, "班级信息", cellStyleMap, exportFields, data);
            ExcelCourseExportService.exportExcel(request, response, workbook, fileName);
            log.debug("export course info success");
        } catch (Exception e) {
            log.error("export course info catch exception:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "导出数据处理异常");
        } finally {
            try {
                workbook.close();
            } catch (IOException var19) {
                log.error("close workbook catch error:", var19);
            }
        }
    }
}
