package com.baijia.tianxiao.biz.dashboard.service.impl;

import com.baijia.tianxiao.biz.dashboard.dto.unKexiao.constant.StatisticsScaleEnum;
import com.baijia.tianxiao.biz.dashboard.dto.unKexiao.reponse.*;
import com.baijia.tianxiao.biz.dashboard.dto.unKexiao.request.ClassUnKexiaoListParam;
import com.baijia.tianxiao.biz.dashboard.dto.unKexiao.request.OrgUnKexiaoListParam;
import com.baijia.tianxiao.biz.dashboard.dto.unKexiao.request.StudentUnKexiaoListParam;
import com.baijia.tianxiao.biz.dashboard.service.DashboardUnKexiaoService;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.dao.*;
import com.baijia.tianxiao.dal.org.po.*;
import com.baijia.tianxiao.enums.StudentCourseStatus;
import com.baijia.tianxiao.sal.common.api.KexiaoApiService;
import com.baijia.tianxiao.sal.common.dto.kexiao.KexiaoStudentStat;
import com.baijia.tianxiao.sal.kexiao.dto.LessonKexiaoStatisticsDto;
import com.baijia.tianxiao.sal.kexiao.service.KexiaoStatisticsService;
import com.baijia.tianxiao.sqlbuilder.bean.Order;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author : fulinyu
 * @title : DashboardUnKexiaoServiceImpl
 * @description : 待消统计
 * @date : 2017年8月22日
 */


//总体思路
//按校区/班级/学员维度统计:
// 1.报名[次数,小时数lesson_count及对应金额 3项之和: pay_price+student_pay_price+transfer_purchase_money]; charge_unit :1按次  2按小时
// 2.退费[次数,小时数 refund_lesson_count 及对应金额refund_price, charge_unit :1按次  4 按分钟];
// 3.排课[次数,小时数及对应金额  lesson_duration 该节课的时长(单位:分钟)]
// 4.转班[转出] lessonCout 数量; lessonMoney 金额; charge_unit 按次数,按分钟   只计算转出;转入走报名
// 超排课 见表org_student_course_lesson ,字段lesson_type:1 正常 ,2 超排(赠送)
//待消=报名+超排(赠送)-退费-转出-已消
//报名要判断订单是否有效 lesson_count>0,无效单子以后的数据全部置0
//排课:tx_signup_course_lesson  对于小时课来说,可能一个课节lesson_id对应多条课消记录[统计时,粒度无需到每个课节,可直接根据获取的所有记录,逐条累计.因此,可忽略该问题].

@Service
@Slf4j
public class DashboardUnKexiaoServiceImpl implements DashboardUnKexiaoService {
    @Autowired
    private OrgInfoDao orgInfoDao;
    @Autowired
    private OrgCourseDao orgCourseDao;
    @Autowired
    private OrgAccountDao orgAccountDao;
    @Autowired
    private OrgStudentDao orgStudentDao;
    @Autowired
    private OrgSubAccountDao orgSubAccountDao;
    @Autowired
    private OrgStudentCourseDao orgStudentCourseDao;
    @Autowired
    private OrgCourseConsumeRuleDao courseConsumeRuleDao;
    @Autowired
    private KexiaoApiService kexiaoApiService;
    @Autowired
    private KexiaoStatisticsService kexiaoStatisticsService;

    @Override
    public OrgUnKexiaoDto getOrgUnKexiaoStatistics(Long orgId, boolean containSubOrg, Collection<Integer> filterSubOrgTypes) {
        log.info("getOrgUnKexiaoStatistics params : orgId = {}, containSubOrg = {}, filterSubOrgTypes = {}", orgId, containSubOrg, filterSubOrgTypes);
        OrgUnKexiaoListParam param = new OrgUnKexiaoListParam();
        param.setOrgId(orgId);
        param.setContainSubOrg(containSubOrg);
        param.setFilterSubOrgTypes(filterSubOrgTypes);
        OrgInfo orgInfo = orgInfoDao.getOrgInfo(orgId.intValue(), "id", "orgId", "shortName");
        Long orgNumber = this.orgAccountDao.getOrgNumbersByOrgIds(Sets.newHashSet(orgId)).get(orgId);
        return OrgUnKexiaoDto.from(orgInfo, orgNumber, this.getOrgUnKexiaoTotal(param));
    }

    @Override
    public List<OrgUnKexiaoDto> getOrgUnKexiaoList(OrgUnKexiaoListParam param) {
        log.info("getOrgUnKexiaoList param : {}", param);
        param.validate();
        List<OrgUnKexiaoDto> result = Lists.newArrayList();
        if (param.isContainSubOrg()) {
            //获取该校区及其子校区的待消
            result = this.getOrgSelfAndSlavesUnKexiaoInfo(param.getOrgId(), param.getFilterSubOrgTypes());
        } else {
            //获取该校区的待消
            OrgUnKexiaoDto orgUnKexiaoDto = this.getOrgSelfUnKexiaoInfo(param.getOrgId());
            if (orgUnKexiaoDto != null) {
                result.add(orgUnKexiaoDto);
            }
        }
        log.info("getOrgUnKexiaoList result.size() : {}", result.size());
        return result;
    }

    private OrgUnKexiaoDto getOrgSelfUnKexiaoInfo(Long orgId) {
        Preconditions.checkArgument(orgId != null, "校区id为空");
        UnKexiaoStatisticsDto unKexiaoStatisticsDto = this.getOrgSelfUnKexiaoTotal(orgId);
        if (UnKexiaoStatisticsDto.isZero(unKexiaoStatisticsDto)) {
            return null;
        }
        OrgInfo orgInfo = orgInfoDao.getOrgInfo(orgId.intValue(), "id", "orgId", "shortName");
        Long orgNumber = orgAccountDao.getOrgNumbersByOrgIds(Sets.newHashSet(orgId)).get(orgId);
        OrgUnKexiaoDto orgUnKexiaoDto = OrgUnKexiaoDto.from(orgInfo, orgNumber, unKexiaoStatisticsDto);
        return orgUnKexiaoDto;
    }

    private List<OrgUnKexiaoDto> getOrgSelfAndSlavesUnKexiaoInfo(Long orgId, Collection<Integer> filterSubOrgTypes) {
        Preconditions.checkArgument(orgId != null, "校区id为空");

        List<Long> allOrgIds = Lists.newArrayList(orgId);
        List<Long> subOrgIds = this.getOrgSlaveIds(orgId, filterSubOrgTypes);
        if (CollectionUtils.isNotEmpty(subOrgIds)) {
            allOrgIds.addAll(subOrgIds);
        }
        Map<Long, UnKexiaoStatisticsDto> orgUnkexiaoStisticsMap = this.getOrgUnKexiaoInfoMap(allOrgIds);

        //过滤待消数据为0的校区
        Map<Long, UnKexiaoStatisticsDto> effectiveOrgUnkexiaoStisticsMap = Maps.newHashMap();
        orgUnkexiaoStisticsMap.forEach((k, e) -> {
            if (!UnKexiaoStatisticsDto.isZero(e)) {
                effectiveOrgUnkexiaoStisticsMap.put(k, e);
            }
        });
        if (MapUtils.isEmpty(effectiveOrgUnkexiaoStisticsMap)) {
            return Collections.emptyList();
        }

        List<Long> orgIds = Lists.newArrayList();
        List<Integer> orgIdsInt = Lists.newArrayList();
        allOrgIds.forEach(o -> {
            if (effectiveOrgUnkexiaoStisticsMap.keySet().contains(o)) {
                orgIdsInt.add(o.intValue());
                orgIds.add(o);
            }
        });

        List<OrgInfo> orgList = orgInfoDao.getOrgInfos(orgIdsInt, "id", "orgId", "shortName");
        Map<Long, OrgInfo> orgInfoMap = orgList.stream().collect(Collectors.toMap(o -> o.getOrgId().longValue(), o -> o));
        Map<Long, Long> orgIdNumberMap = this.orgAccountDao.getOrgNumbersByOrgIds(allOrgIds);

        return OrgUnKexiaoDto.listFrom(orgIds, orgIdNumberMap, orgInfoMap, orgUnkexiaoStisticsMap);
    }

    private Map<Long, UnKexiaoStatisticsDto> getOrgUnKexiaoInfoMap(List<Long> orgIds) {
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(orgIds), "校区ids为空");

        Map<Long, UnKexiaoStatisticsDto> dtoMap = Maps.newHashMap();
        Map<Long, LessonKexiaoStatisticsDto> signupMap = this.kexiaoStatisticsService.getSignupCourseStatisticsMapOfOrg(orgIds);
        if (LessonKexiaoStatisticsDto.isZeroMap(signupMap)) {
            for (Long orgId : orgIds) {
                dtoMap.put(orgId, UnKexiaoStatisticsDto.createWithZeroValue());
            }
        } else {
            Map<Long, LessonKexiaoStatisticsDto> freeLessonMap = this.kexiaoStatisticsService.getLessonFreeMapOfOrg(orgIds);
            Map<Long, LessonKexiaoStatisticsDto> kexiaoFinishedMap = this.kexiaoStatisticsService.getKexiaoFinishedMapOfOrg(orgIds);
            Map<Long, LessonKexiaoStatisticsDto> refundMap = this.kexiaoStatisticsService.getRefundMapOfOrg(orgIds);
            Map<Long, LessonKexiaoStatisticsDto> transferMap = this.kexiaoStatisticsService.getTransferMapOfOrg(orgIds);
            for (Long id : orgIds) {
                LessonKexiaoStatisticsDto signupStatisticsDto = signupMap.get(id);
                LessonKexiaoStatisticsDto freeLessonStatisticsDto = freeLessonMap.get(id);
                LessonKexiaoStatisticsDto kexiaoFinishedStatisticsDto = kexiaoFinishedMap.get(id);
                LessonKexiaoStatisticsDto refundStatisticsDto = refundMap.get(id);
                LessonKexiaoStatisticsDto transferStatisticsDto = transferMap.get(id);
                UnKexiaoStatisticsDto dto = UnKexiaoStatisticsDto.fromLessonKexiaoStatisticsDto(signupStatisticsDto, freeLessonStatisticsDto, kexiaoFinishedStatisticsDto, refundStatisticsDto, transferStatisticsDto);
                dtoMap.put(id, dto);
            }
        }
        return dtoMap;
    }

    @Override
    public UnKexiaoStatisticsDto getOrgUnKexiaoTotal(OrgUnKexiaoListParam param) {
        log.info("getOrgUnKexiaoTotal param : {}", param);
        param.validate();
        UnKexiaoStatisticsDto result = null;
        if (param.isContainSubOrg()) {
            result = this.getOrgSelfAndSlavesUnKexiaoTotal(param.getOrgId(), param.getFilterSubOrgTypes());
        } else {
            //获取该校区的待消
            result = this.getOrgSelfUnKexiaoTotal(param.getOrgId());
        }
        log.info("getOrgUnKexiaoTotal result : {}", result);
        return result;
    }


    private UnKexiaoStatisticsDto getOrgSelfUnKexiaoTotal(Long orgId) {
        Preconditions.checkArgument(orgId != null, "校区id为空");
        UnKexiaoStatisticsDto result = this.getUnKexiaoSumOfOrgs(Sets.newHashSet(orgId));
        return result;
    }

    private List<Long> getOrgSlaveIds(Long orgId, Collection<Integer> filterSubOrgTypes) {
        if (orgId == null) {
            return Collections.emptyList();
        }
        List<Long> orgIds = Lists.newArrayList();
        List<OrgSubAccount> subAccounts = orgSubAccountDao.getSlavesByMasterOrgId(orgId.intValue(), null);
        if (CollectionUtils.isNotEmpty(subAccounts)) {
            for (OrgSubAccount account : subAccounts) {
                //去除过滤类型的子校区
                if (!(CollectionUtils.isNotEmpty(filterSubOrgTypes) && filterSubOrgTypes.contains(account.getAccountType().intValue()))) {
                    orgIds.add(account.getOrgId().longValue());
                }
            }
        }
        return orgIds;
    }

    private UnKexiaoStatisticsDto getOrgSelfAndSlavesUnKexiaoTotal(Long orgId, Collection<Integer> filterSubOrgTypes) {
        Preconditions.checkArgument(orgId != null, "校区id为空");
        List<Long> allOrgIds = Lists.newArrayList(orgId);
        List<Long> subOrgIds = this.getOrgSlaveIds(orgId, filterSubOrgTypes);
        if (CollectionUtils.isNotEmpty(subOrgIds)) {
            allOrgIds.addAll(subOrgIds);
        }
        UnKexiaoStatisticsDto result = this.getUnKexiaoSumOfOrgs(allOrgIds);
        return result;
    }


    private UnKexiaoStatisticsDto getUnKexiaoSumOfOrgs(Collection<Long> orgIds) {
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(orgIds), "校区ids为空");
        UnKexiaoStatisticsDto result = null;

        Map<Long, LessonKexiaoStatisticsDto> signupMap = this.kexiaoStatisticsService.getSignupCourseStatisticsMapOfOrg(orgIds);
        if (LessonKexiaoStatisticsDto.isZeroMap(signupMap)) {
            result = UnKexiaoStatisticsDto.createWithZeroValue();
        } else {

            LessonKexiaoStatisticsDto signupStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto freeLessonStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto kexiaoFinishedStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto refundStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto transferStatisticsTotalDto = new LessonKexiaoStatisticsDto();

            Map<Long, LessonKexiaoStatisticsDto> freeLessonMap = this.kexiaoStatisticsService.getLessonFreeMapOfOrg(orgIds);
            Map<Long, LessonKexiaoStatisticsDto> kexiaoFinishedMap = this.kexiaoStatisticsService.getKexiaoFinishedMapOfOrg(orgIds);
            Map<Long, LessonKexiaoStatisticsDto> refundMap = this.kexiaoStatisticsService.getRefundMapOfOrg(orgIds);
            Map<Long, LessonKexiaoStatisticsDto> transferMap = this.kexiaoStatisticsService.getTransferMapOfOrg(orgIds);

            for (Long id : orgIds) {
                LessonKexiaoStatisticsDto signupStatisticsDto = signupMap.get(id);
                signupStatisticsTotalDto.add(signupStatisticsDto);
                LessonKexiaoStatisticsDto freeLessonStatisticsDto = freeLessonMap.get(id);
                freeLessonStatisticsTotalDto.add(freeLessonStatisticsDto);
                LessonKexiaoStatisticsDto kexiaoFinishedStatisticsDto = kexiaoFinishedMap.get(id);
                kexiaoFinishedStatisticsTotalDto.add(kexiaoFinishedStatisticsDto);
                LessonKexiaoStatisticsDto refundStatisticsDto = refundMap.get(id);
                refundStatisticsTotalDto.add(refundStatisticsDto);
                LessonKexiaoStatisticsDto transferStatisticsDto = transferMap.get(id);
                transferStatisticsTotalDto.add(transferStatisticsDto);
            }
            result = UnKexiaoStatisticsDto.fromLessonKexiaoStatisticsDto(signupStatisticsTotalDto, freeLessonStatisticsTotalDto, kexiaoFinishedStatisticsTotalDto, refundStatisticsTotalDto, transferStatisticsTotalDto);
        }
        return result;
    }


    @Override
    public ClassStatisticsDto getClassUnKexiaoStatistics(Long orgId, Long classId) {
        Preconditions.checkArgument(orgId != null, "校区id为空");
        Preconditions.checkArgument(classId != null, "班级id为空");

        OrgCourse orgClass = orgCourseDao.getByCourseId(classId, "id", "name", "courseType", "chargeType", "chargeUnit", "parentId");
        Preconditions.checkArgument(orgClass != null, "班级不存在");
        long courseId = classId;
        if (orgClass.getParentId() != null && orgClass.getParentId() > 0) {
            courseId = orgClass.getParentId();
        }
        OrgCourseConsumeRule kexiaoRule = courseConsumeRuleDao.getRuleByCourseId(orgId, courseId);
        Integer kexiaoRuleInt = (kexiaoRule == null) ? null : kexiaoRule.getRuleValue();
        Integer studentCount = orgStudentCourseDao.countStudents(orgId, classId, StudentCourseStatus.NORMAL.getCode());
        UnKexiaoStatisticsDto unKexiaoStatisticsDto = this.getClassUnKexiaoInfo(classId);
        return ClassStatisticsDto.from(orgClass, kexiaoRuleInt, studentCount, unKexiaoStatisticsDto);
    }

    @Override
    public List<ClassUnKexiaoDto> getClassUnKexiaoList(ClassUnKexiaoListParam param) {
        log.info("getClassUnKexiaoList  params : {}", param);
        param.validate();
        OrgAccount org = orgAccountDao.getById(param.getOrgId());
        if (org == null) {
            return Collections.emptyList();
        } else {
            List<Long> classIds = Lists.newArrayList();
            if (param.getScale().intValue() == StatisticsScaleEnum.ORG.getCode()) {
                classIds = orgCourseDao.listCourseIdIgnoreStatus(org.getNumber().longValue(), null, param.getKeyword(), CourseTypeEnum.getCourseTypeList(param.getCourseType()), param.getChargeType(), param.getChargeUnit(), null);
            } else if (param.getScale().intValue() == StatisticsScaleEnum.STUDENT.getCode()) {
                List<OrgStudentCourse> studentClass = orgStudentCourseDao.listByStudentUserId(param.getOrgId(), param.getStudentUserId(), null, null);
                classIds = studentClass.stream().map(OrgStudentCourse::getCourseId).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(classIds)) {
                    classIds = orgCourseDao.listCourseIdIgnoreStatus(org.getNumber().longValue(), classIds, param.getKeyword(), CourseTypeEnum.getCourseTypeList(param.getCourseType()), param.getChargeType(), param.getChargeUnit(), null);
                }
            }
            if (CollectionUtils.isEmpty(classIds)) {
                return Collections.emptyList();
            } else {
                //1.班级的待消情况
                Map<Long, UnKexiaoStatisticsDto> classUnkexiaoStisticsMap = Maps.newHashMap();
                if (param.getScale().intValue() == StatisticsScaleEnum.ORG.getCode()) {
                    classUnkexiaoStisticsMap = this.getClassUnKexiaoInfoMap(classIds);
                } else {
                    classUnkexiaoStisticsMap = this.getStudentClassUnKexiaoInfoMap(param.getStudentUserId(), classIds);
                }
                //过滤待消数据为0的班级
                Map<Long, UnKexiaoStatisticsDto> effectiveClassUnkexiaoStisticsMap = Maps.newHashMap();
                classUnkexiaoStisticsMap.forEach((k, e) -> {
                    if (!UnKexiaoStatisticsDto.isZero(e)) {
                        effectiveClassUnkexiaoStisticsMap.put(k, e);
                    }
                });
                if (MapUtils.isEmpty(effectiveClassUnkexiaoStisticsMap)) {
                    return Collections.emptyList();
                }

                //2.班级的基本信息:按创建时间倒序排序
                List<OrgCourse> orgClassList = this.orgCourseDao.getClassList(effectiveClassUnkexiaoStisticsMap.keySet(), Lists.newArrayList(Order.desc("id")), param.getPageDto(), "id", "name", "courseType", "chargeType", "chargeUnit", "isDel");
                Map<Long, OrgCourse> orgClassMap = orgClassList.stream().collect(Collectors.toMap(o -> o.getId(), o -> o));
                classIds = orgClassList.stream().map(OrgCourse::getId).collect(Collectors.toList());

                return ClassUnKexiaoDto.listFrom(classIds, orgClassMap, effectiveClassUnkexiaoStisticsMap);
            }
        }
    }


    @Override
    public UnKexiaoStatisticsDto getClassUnKexiaoTotal(ClassUnKexiaoListParam param) {
        param.validate();
        //如果,统计一个校区的班级&&查询条件为空,则求该校区的待消汇总数据
        //如果,统计一个学员的班级&&查询条件为空,则求该学员的待消汇总数据
        //否则,根据查询条件查出班级,计算这些班级的待消汇总数据
        if (param.getScale().intValue() == StatisticsScaleEnum.ORG.getCode() && param.isEmptyConditionQuery()) {
            return this.getOrgSelfUnKexiaoInfo(param.getOrgId());
        }
        if (param.getScale().intValue() == StatisticsScaleEnum.STUDENT.getCode() && param.isEmptyConditionQuery()) {
            return this.getStudentUnKexiaoInfo(param.getStudentUserId());
        }
        UnKexiaoStatisticsDto result = null;
        OrgAccount org = orgAccountDao.getById(param.getOrgId());
        if (org == null) {
            result = UnKexiaoStatisticsDto.createWithZeroValue();
        } else {
            List<Long> classIds = Lists.newArrayList();
            if (param.getScale().intValue() == StatisticsScaleEnum.ORG.getCode()) {
                classIds = orgCourseDao.listCourseIdIgnoreStatus(org.getNumber().longValue(), null, param.getKeyword(), CourseTypeEnum.getCourseTypeList(param.getCourseType()), param.getChargeType(), param.getChargeUnit(), null);
            } else if (param.getScale().intValue() == StatisticsScaleEnum.STUDENT.getCode()) {
                List<OrgStudentCourse> studentClass = orgStudentCourseDao.listByStudentUserId(param.getOrgId(), param.getStudentUserId(), null, null);
                classIds = studentClass.stream().map(OrgStudentCourse::getCourseId).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(classIds)) {
                    classIds = orgCourseDao.listCourseIdIgnoreStatus(org.getNumber().longValue(), classIds, param.getKeyword(), CourseTypeEnum.getCourseTypeList(param.getCourseType()), param.getChargeType(), param.getChargeUnit(), null);
                }
            }
            if (CollectionUtils.isEmpty(classIds)) {
                result = UnKexiaoStatisticsDto.createWithZeroValue();
            } else {
                if (param.getScale().intValue() == StatisticsScaleEnum.ORG.getCode()) {
                    result = this.getUnKexiaoSumOfClasses(classIds);
                } else {
                    result = this.getUnKexiaoSumOfStudentClasses(param.getStudentUserId(), classIds);
                }
            }
        }
        return result;
    }

    private UnKexiaoStatisticsDto getClassUnKexiaoInfo(Long classId) {
        Preconditions.checkArgument(classId != null, "班级id为空");
        return this.getUnKexiaoSumOfClasses(Sets.newHashSet(classId));
    }

    private UnKexiaoStatisticsDto getUnKexiaoSumOfClasses(Collection<Long> classIds) {
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(classIds), "班级ids为空");
        UnKexiaoStatisticsDto result = null;

        Map<Long, LessonKexiaoStatisticsDto> signupMap = this.kexiaoStatisticsService.getSignupCourseStatisticsMapOfClass(classIds);
        if (LessonKexiaoStatisticsDto.isZeroMap(signupMap)) {
            result = UnKexiaoStatisticsDto.createWithZeroValue();
        } else {

            LessonKexiaoStatisticsDto signupStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto freeLessonStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto kexiaoFinishedStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto refundStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto transferStatisticsTotalDto = new LessonKexiaoStatisticsDto();

            Map<Long, LessonKexiaoStatisticsDto> freeLessonMap = this.kexiaoStatisticsService.getLessonFreeMapOfClass(classIds);
            Map<Long, LessonKexiaoStatisticsDto> kexiaoFinishedMap = this.kexiaoStatisticsService.getKexiaoFinishedMapOfClass(classIds);
            Map<Long, LessonKexiaoStatisticsDto> refundMap = this.kexiaoStatisticsService.getRefundMapOfClass(classIds);
            Map<Long, LessonKexiaoStatisticsDto> transferMap = this.kexiaoStatisticsService.getTransferMapOfClass(classIds);

            for (Long id : classIds) {
                LessonKexiaoStatisticsDto signupStatisticsDto = signupMap.get(id);
                signupStatisticsTotalDto.add(signupStatisticsDto);
                LessonKexiaoStatisticsDto freeLessonStatisticsDto = freeLessonMap.get(id);
                freeLessonStatisticsTotalDto.add(freeLessonStatisticsDto);
                LessonKexiaoStatisticsDto kexiaoFinishedStatisticsDto = kexiaoFinishedMap.get(id);
                kexiaoFinishedStatisticsTotalDto.add(kexiaoFinishedStatisticsDto);
                LessonKexiaoStatisticsDto refundStatisticsDto = refundMap.get(id);
                refundStatisticsTotalDto.add(refundStatisticsDto);
                LessonKexiaoStatisticsDto transferStatisticsDto = transferMap.get(id);
                transferStatisticsTotalDto.add(transferStatisticsDto);
            }
            result = UnKexiaoStatisticsDto.fromLessonKexiaoStatisticsDto(signupStatisticsTotalDto, freeLessonStatisticsTotalDto, kexiaoFinishedStatisticsTotalDto, refundStatisticsTotalDto, transferStatisticsTotalDto);
        }
        return result;
    }

    private UnKexiaoStatisticsDto getUnKexiaoSumOfStudentClasses(Long userId, Collection<Long> classIds) {
        Preconditions.checkArgument(userId != null, "学员id为空");
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(classIds), "班级ids为空");
        UnKexiaoStatisticsDto result = null;

        Map<Long, LessonKexiaoStatisticsDto> signupMap = this.kexiaoStatisticsService.getSignupCourseStatisticsMapOfStudentClass(userId, classIds);
        if (LessonKexiaoStatisticsDto.isZeroMap(signupMap)) {
            result = UnKexiaoStatisticsDto.createWithZeroValue();
        } else {

            LessonKexiaoStatisticsDto signupStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto freeLessonStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto kexiaoFinishedStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto refundStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto transferStatisticsTotalDto = new LessonKexiaoStatisticsDto();

            Map<Long, LessonKexiaoStatisticsDto> freeLessonMap = this.kexiaoStatisticsService.getLessonFreeMapOfStudentClass(userId, classIds);
            Map<Long, LessonKexiaoStatisticsDto> kexiaoFinishedMap = this.kexiaoStatisticsService.getKexiaoFinishedMapOfStudentClass(userId, classIds);
            Map<Long, LessonKexiaoStatisticsDto> refundMap = this.kexiaoStatisticsService.getRefundMapOfStudentClass(userId, classIds);
            Map<Long, LessonKexiaoStatisticsDto> transferMap = this.kexiaoStatisticsService.getTransferMapOfStudentClass(userId, classIds);

            for (Long id : classIds) {
                LessonKexiaoStatisticsDto signupStatisticsDto = signupMap.get(id);
                signupStatisticsTotalDto.add(signupStatisticsDto);
                LessonKexiaoStatisticsDto freeLessonStatisticsDto = freeLessonMap.get(id);
                freeLessonStatisticsTotalDto.add(freeLessonStatisticsDto);
                LessonKexiaoStatisticsDto kexiaoFinishedStatisticsDto = kexiaoFinishedMap.get(id);
                kexiaoFinishedStatisticsTotalDto.add(kexiaoFinishedStatisticsDto);
                LessonKexiaoStatisticsDto refundStatisticsDto = refundMap.get(id);
                refundStatisticsTotalDto.add(refundStatisticsDto);
                LessonKexiaoStatisticsDto transferStatisticsDto = transferMap.get(id);
                transferStatisticsTotalDto.add(transferStatisticsDto);
            }
            result = UnKexiaoStatisticsDto.fromLessonKexiaoStatisticsDto(signupStatisticsTotalDto, freeLessonStatisticsTotalDto, kexiaoFinishedStatisticsTotalDto, refundStatisticsTotalDto, transferStatisticsTotalDto);
        }
        return result;
    }


    private Map<Long, UnKexiaoStatisticsDto> getClassUnKexiaoInfoMap(List<Long> classIds) {
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(classIds), "班级ids为空");

        Map<Long, UnKexiaoStatisticsDto> dtoMap = Maps.newHashMap();
        Map<Long, LessonKexiaoStatisticsDto> signupMap = this.kexiaoStatisticsService.getSignupCourseStatisticsMapOfClass(classIds);
        if (LessonKexiaoStatisticsDto.isZeroMap(signupMap)) {
            for (Long orgId : classIds) {
                dtoMap.put(orgId, UnKexiaoStatisticsDto.createWithZeroValue());
            }
        } else {
            Map<Long, LessonKexiaoStatisticsDto> freeLessonMap = this.kexiaoStatisticsService.getLessonFreeMapOfClass(classIds);
            Map<Long, LessonKexiaoStatisticsDto> kexiaoFinishedMap = this.kexiaoStatisticsService.getKexiaoFinishedMapOfClass(classIds);
            Map<Long, LessonKexiaoStatisticsDto> refundMap = this.kexiaoStatisticsService.getRefundMapOfClass(classIds);
            Map<Long, LessonKexiaoStatisticsDto> transferMap = this.kexiaoStatisticsService.getTransferMapOfClass(classIds);
            for (Long id : classIds) {
                LessonKexiaoStatisticsDto signupStatisticsDto = signupMap.get(id);
                LessonKexiaoStatisticsDto freeLessonStatisticsDto = freeLessonMap.get(id);
                LessonKexiaoStatisticsDto kexiaoFinishedStatisticsDto = kexiaoFinishedMap.get(id);
                LessonKexiaoStatisticsDto refundStatisticsDto = refundMap.get(id);
                LessonKexiaoStatisticsDto transferStatisticsDto = transferMap.get(id);
                UnKexiaoStatisticsDto dto = UnKexiaoStatisticsDto.fromLessonKexiaoStatisticsDto(signupStatisticsDto, freeLessonStatisticsDto, kexiaoFinishedStatisticsDto, refundStatisticsDto, transferStatisticsDto);
                dtoMap.put(id, dto);
            }
        }
        return dtoMap;
    }

    /**
     * 学员在班级的待消情况
     *
     * @param userId
     * @param classIds
     * @return
     */
    private Map<Long, UnKexiaoStatisticsDto> getStudentClassUnKexiaoInfoMap(Long userId, List<Long> classIds) {
        Preconditions.checkArgument(userId != null, "学员id为空");
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(classIds), "班级ids为空");

        Map<Long, UnKexiaoStatisticsDto> dtoMap = Maps.newHashMap();
        Map<Long, LessonKexiaoStatisticsDto> signupMap = this.kexiaoStatisticsService.getSignupCourseStatisticsMapOfStudentClass(userId, classIds);
        if (LessonKexiaoStatisticsDto.isZeroMap(signupMap)) {
            for (Long orgId : classIds) {
                dtoMap.put(orgId, UnKexiaoStatisticsDto.createWithZeroValue());
            }
        } else {
            Map<Long, LessonKexiaoStatisticsDto> freeLessonMap = this.kexiaoStatisticsService.getLessonFreeMapOfStudentClass(userId, classIds);
            Map<Long, LessonKexiaoStatisticsDto> kexiaoFinishedMap = this.kexiaoStatisticsService.getKexiaoFinishedMapOfStudentClass(userId, classIds);
            Map<Long, LessonKexiaoStatisticsDto> refundMap = this.kexiaoStatisticsService.getRefundMapOfStudentClass(userId, classIds);
            Map<Long, LessonKexiaoStatisticsDto> transferMap = this.kexiaoStatisticsService.getTransferMapOfStudentClass(userId, classIds);
            for (Long id : classIds) {
                LessonKexiaoStatisticsDto signupStatisticsDto = signupMap.get(id);
                LessonKexiaoStatisticsDto freeLessonStatisticsDto = freeLessonMap.get(id);
                LessonKexiaoStatisticsDto kexiaoFinishedStatisticsDto = kexiaoFinishedMap.get(id);
                LessonKexiaoStatisticsDto refundStatisticsDto = refundMap.get(id);
                LessonKexiaoStatisticsDto transferStatisticsDto = transferMap.get(id);
                UnKexiaoStatisticsDto dto = UnKexiaoStatisticsDto.fromLessonKexiaoStatisticsDto(signupStatisticsDto, freeLessonStatisticsDto, kexiaoFinishedStatisticsDto, refundStatisticsDto, transferStatisticsDto);
                dtoMap.put(id, dto);
            }
        }
        return dtoMap;
    }


    @Override
    public StudentStatisticsDto getStudentUnKexiaoStatistics(Long orgId, Long studentUserId) {
        Preconditions.checkArgument(orgId != null, "校区id为空");
        Preconditions.checkArgument(studentUserId != null, "学员id为空");

        OrgStudent student = orgStudentDao.getStudentByUserId(orgId, studentUserId, "id", "userId", "name");
        Preconditions.checkArgument(student != null, "学员不存在");
        //在读班级数
        Integer classCount = orgStudentCourseDao.getStudentSignupCourseCount(orgId, studentUserId, StudentCourseStatus.NORMAL.getCode());
        //缴费总额,剩余学费
        Long payMoneyLong = 0l;
        Long remainingMoneyLong = 0l;
        KexiaoStudentStat kexiaoStudentStat = this.kexiaoApiService.queryUserKexiaoStat(orgId, Sets.newHashSet(studentUserId)).get(studentUserId);
        if (kexiaoStudentStat != null) {
            payMoneyLong = kexiaoStudentStat.getTotalAmount();
            remainingMoneyLong = kexiaoStudentStat.getLeftAmount();
        }
        //待消数据
        UnKexiaoStatisticsDto unKexiaoStatisticsDto = this.getStudentUnKexiaoInfo(studentUserId);
        return StudentStatisticsDto.from(student, classCount, payMoneyLong, remainingMoneyLong, unKexiaoStatisticsDto);
    }

    private UnKexiaoStatisticsDto getStudentUnKexiaoInfo(Long userId) {
        Preconditions.checkArgument(userId != null, "学员id为空");
        return this.getUnKexiaoSumOfStudents(Sets.newHashSet(userId));
    }

    private UnKexiaoStatisticsDto getUnKexiaoSumOfStudents(Collection<Long> userIds) {
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(userIds), "学员ids为空");
        UnKexiaoStatisticsDto result = null;

        Map<Long, LessonKexiaoStatisticsDto> signupMap = this.kexiaoStatisticsService.getSignupCourseStatisticsMapOfStudent(userIds);
        if (LessonKexiaoStatisticsDto.isZeroMap(signupMap)) {
            result = UnKexiaoStatisticsDto.createWithZeroValue();
        } else {

            LessonKexiaoStatisticsDto signupStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto freeLessonStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto kexiaoFinishedStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto refundStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto transferStatisticsTotalDto = new LessonKexiaoStatisticsDto();

            Map<Long, LessonKexiaoStatisticsDto> freeLessonMap = this.kexiaoStatisticsService.getLessonFreeMapOfStudent(userIds);
            Map<Long, LessonKexiaoStatisticsDto> kexiaoFinishedMap = this.kexiaoStatisticsService.getKexiaoFinishedMapOfStudent(userIds);
            Map<Long, LessonKexiaoStatisticsDto> refundMap = this.kexiaoStatisticsService.getRefundMapOfStudent(userIds);
            Map<Long, LessonKexiaoStatisticsDto> transferMap = this.kexiaoStatisticsService.getTransferMapOfStudent(userIds);

            for (Long id : userIds) {
                LessonKexiaoStatisticsDto signupStatisticsDto = signupMap.get(id);
                signupStatisticsTotalDto.add(signupStatisticsDto);
                LessonKexiaoStatisticsDto freeLessonStatisticsDto = freeLessonMap.get(id);
                freeLessonStatisticsTotalDto.add(freeLessonStatisticsDto);
                LessonKexiaoStatisticsDto kexiaoFinishedStatisticsDto = kexiaoFinishedMap.get(id);
                kexiaoFinishedStatisticsTotalDto.add(kexiaoFinishedStatisticsDto);
                LessonKexiaoStatisticsDto refundStatisticsDto = refundMap.get(id);
                refundStatisticsTotalDto.add(refundStatisticsDto);
                LessonKexiaoStatisticsDto transferStatisticsDto = transferMap.get(id);
                transferStatisticsTotalDto.add(transferStatisticsDto);
            }
            result = UnKexiaoStatisticsDto.fromLessonKexiaoStatisticsDto(signupStatisticsTotalDto, freeLessonStatisticsTotalDto, kexiaoFinishedStatisticsTotalDto, refundStatisticsTotalDto, transferStatisticsTotalDto);
        }
        return result;
    }

    private UnKexiaoStatisticsDto getUnKexiaoSumOfClassStudents(Long classId, Collection<Long> userIds) {
        Preconditions.checkArgument(classId != null, "班级id为空");
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(userIds), "学员ids为空");
        UnKexiaoStatisticsDto result = null;

        Map<Long, LessonKexiaoStatisticsDto> signupMap = this.kexiaoStatisticsService.getSignupCourseStatisticsMapOfClassStudent(classId, userIds);
        if (LessonKexiaoStatisticsDto.isZeroMap(signupMap)) {
            result = UnKexiaoStatisticsDto.createWithZeroValue();
        } else {

            LessonKexiaoStatisticsDto signupStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto freeLessonStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto kexiaoFinishedStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto refundStatisticsTotalDto = new LessonKexiaoStatisticsDto();
            LessonKexiaoStatisticsDto transferStatisticsTotalDto = new LessonKexiaoStatisticsDto();

            Map<Long, LessonKexiaoStatisticsDto> freeLessonMap = this.kexiaoStatisticsService.getLessonFreeMapOfClassStudent(classId, userIds);
            Map<Long, LessonKexiaoStatisticsDto> kexiaoFinishedMap = this.kexiaoStatisticsService.getKexiaoFinishedMapOfClassStudent(classId, userIds);
            Map<Long, LessonKexiaoStatisticsDto> refundMap = this.kexiaoStatisticsService.getRefundMapOfClassStudent(classId, userIds);
            Map<Long, LessonKexiaoStatisticsDto> transferMap = this.kexiaoStatisticsService.getTransferMapOfClassStudent(classId, userIds);

            for (Long id : userIds) {
                LessonKexiaoStatisticsDto signupStatisticsDto = signupMap.get(id);
                signupStatisticsTotalDto.add(signupStatisticsDto);
                LessonKexiaoStatisticsDto freeLessonStatisticsDto = freeLessonMap.get(id);
                freeLessonStatisticsTotalDto.add(freeLessonStatisticsDto);
                LessonKexiaoStatisticsDto kexiaoFinishedStatisticsDto = kexiaoFinishedMap.get(id);
                kexiaoFinishedStatisticsTotalDto.add(kexiaoFinishedStatisticsDto);
                LessonKexiaoStatisticsDto refundStatisticsDto = refundMap.get(id);
                refundStatisticsTotalDto.add(refundStatisticsDto);
                LessonKexiaoStatisticsDto transferStatisticsDto = transferMap.get(id);
                transferStatisticsTotalDto.add(transferStatisticsDto);
            }
            result = UnKexiaoStatisticsDto.fromLessonKexiaoStatisticsDto(signupStatisticsTotalDto, freeLessonStatisticsTotalDto, kexiaoFinishedStatisticsTotalDto, refundStatisticsTotalDto, transferStatisticsTotalDto);
        }
        return result;
    }


    @Override
    public List<StudentUnKexiaoDto> getStudentUnKexiaoList(StudentUnKexiaoListParam param) {
        log.info("getStudentUnKexiaoList params : {}", param);
        param.validate();

        OrgAccount org = orgAccountDao.getById(param.getOrgId());
        if (org == null) {
            return Collections.emptyList();
        } else {
            List<Long> studentUserIds = Lists.newArrayList();
            if (param.getScale().intValue() == StatisticsScaleEnum.ORG.getCode()) {
                studentUserIds = orgStudentDao.listUserIdIgnoreStatus(param.getOrgId(), null, param.getKeyword(), null);
            } else if (param.getScale().intValue() == StatisticsScaleEnum.CLASS.getCode()) {
                //班级学员
                List<OrgStudentCourse> classStudent = orgStudentCourseDao.listByCourseId(param.getOrgId(), param.getClassId(), null, null);
                studentUserIds = classStudent.stream().map(OrgStudentCourse::getUserId).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(studentUserIds)) {
                    studentUserIds = orgStudentDao.listUserIdIgnoreStatus(param.getOrgId(), studentUserIds, param.getKeyword(), null);
                }
            }
            if (CollectionUtils.isEmpty(studentUserIds)) {
                return Collections.emptyList();
            }


            //1.学员的待消数据
            Map<Long, UnKexiaoStatisticsDto> studentUnkexiaoStisticsMap = Maps.newHashMap();
            if (param.getScale().intValue() == StatisticsScaleEnum.ORG.getCode()) {
                studentUnkexiaoStisticsMap = this.getStudentUnKexiaoInfoMap(studentUserIds);
            } else {
                studentUnkexiaoStisticsMap = this.getClassStudentUnKexiaoInfoMap(param.getClassId(), studentUserIds);
            }
            //过滤待消数据为0的学员
            Map<Long, UnKexiaoStatisticsDto> effectiveUnkexiaoStisticsMap = Maps.newHashMap();
            studentUnkexiaoStisticsMap.forEach((k, e) -> {
                if (!UnKexiaoStatisticsDto.isZero(e)) {
                    effectiveUnkexiaoStisticsMap.put(k, e);
                }
            });
            if (MapUtils.isEmpty(effectiveUnkexiaoStisticsMap)) {
                return Collections.emptyList();
            }

            List<OrgStudent> orgStudentList = this.orgStudentDao.getStudentLists(effectiveUnkexiaoStisticsMap.keySet(), Lists.newArrayList(Order.desc("id")), param.getPageDto(), "id", "userId", "name", "delStatus");
            Map<Long, OrgStudent> orgStudentMap = orgStudentList.stream().collect(Collectors.toMap(o -> o.getUserId(), o -> o));
            studentUserIds = orgStudentList.stream().map(OrgStudent::getUserId).collect(Collectors.toList());

            return StudentUnKexiaoDto.listFrom(studentUserIds, orgStudentMap, effectiveUnkexiaoStisticsMap);
        }
    }

    private Map<Long, UnKexiaoStatisticsDto> getStudentUnKexiaoInfoMap(List<Long> userIds) {
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(userIds), "学员ids为空");

        Map<Long, UnKexiaoStatisticsDto> dtoMap = Maps.newHashMap();
        Map<Long, LessonKexiaoStatisticsDto> signupMap = this.kexiaoStatisticsService.getSignupCourseStatisticsMapOfStudent(userIds);
        if (LessonKexiaoStatisticsDto.isZeroMap(signupMap)) {
            for (Long orgId : userIds) {
                dtoMap.put(orgId, UnKexiaoStatisticsDto.createWithZeroValue());
            }
        } else {
            Map<Long, LessonKexiaoStatisticsDto> freeLessonMap = this.kexiaoStatisticsService.getLessonFreeMapOfStudent(userIds);
            Map<Long, LessonKexiaoStatisticsDto> kexiaoFinishedMap = this.kexiaoStatisticsService.getKexiaoFinishedMapOfStudent(userIds);
            Map<Long, LessonKexiaoStatisticsDto> refundMap = this.kexiaoStatisticsService.getRefundMapOfStudent(userIds);
            Map<Long, LessonKexiaoStatisticsDto> transferMap = this.kexiaoStatisticsService.getTransferMapOfStudent(userIds);
            for (Long id : userIds) {
                LessonKexiaoStatisticsDto signupStatisticsDto = signupMap.get(id);
                LessonKexiaoStatisticsDto freeLessonStatisticsDto = freeLessonMap.get(id);
                LessonKexiaoStatisticsDto kexiaoFinishedStatisticsDto = kexiaoFinishedMap.get(id);
                LessonKexiaoStatisticsDto refundStatisticsDto = refundMap.get(id);
                LessonKexiaoStatisticsDto transferStatisticsDto = transferMap.get(id);
                UnKexiaoStatisticsDto dto = UnKexiaoStatisticsDto.fromLessonKexiaoStatisticsDto(signupStatisticsDto, freeLessonStatisticsDto, kexiaoFinishedStatisticsDto, refundStatisticsDto, transferStatisticsDto);
                dtoMap.put(id, dto);
            }
        }
        return dtoMap;
    }

    private Map<Long, UnKexiaoStatisticsDto> getClassStudentUnKexiaoInfoMap(Long classId, List<Long> userIds) {
        Preconditions.checkArgument(classId != null, "班级id为空");
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(userIds), "学员ids为空");

        Map<Long, UnKexiaoStatisticsDto> dtoMap = Maps.newHashMap();
        Map<Long, LessonKexiaoStatisticsDto> signupMap = this.kexiaoStatisticsService.getSignupCourseStatisticsMapOfClassStudent(classId, userIds);
        if (LessonKexiaoStatisticsDto.isZeroMap(signupMap)) {
            for (Long orgId : userIds) {
                dtoMap.put(orgId, UnKexiaoStatisticsDto.createWithZeroValue());
            }
        } else {
            Map<Long, LessonKexiaoStatisticsDto> freeLessonMap = this.kexiaoStatisticsService.getLessonFreeMapOfClassStudent(classId, userIds);
            Map<Long, LessonKexiaoStatisticsDto> kexiaoFinishedMap = this.kexiaoStatisticsService.getKexiaoFinishedMapOfClassStudent(classId, userIds);
            Map<Long, LessonKexiaoStatisticsDto> refundMap = this.kexiaoStatisticsService.getRefundMapOfClassStudent(classId, userIds);
            Map<Long, LessonKexiaoStatisticsDto> transferMap = this.kexiaoStatisticsService.getTransferMapOfClassStudent(classId, userIds);
            for (Long id : userIds) {
                LessonKexiaoStatisticsDto signupStatisticsDto = signupMap.get(id);
                LessonKexiaoStatisticsDto freeLessonStatisticsDto = freeLessonMap.get(id);
                LessonKexiaoStatisticsDto kexiaoFinishedStatisticsDto = kexiaoFinishedMap.get(id);
                LessonKexiaoStatisticsDto refundStatisticsDto = refundMap.get(id);
                LessonKexiaoStatisticsDto transferStatisticsDto = transferMap.get(id);
                UnKexiaoStatisticsDto dto = UnKexiaoStatisticsDto.fromLessonKexiaoStatisticsDto(signupStatisticsDto, freeLessonStatisticsDto, kexiaoFinishedStatisticsDto, refundStatisticsDto, transferStatisticsDto);
                dtoMap.put(id, dto);
            }
        }
        return dtoMap;
    }


    @Override
    public UnKexiaoStatisticsDto getStudentUnKexiaoTotal(StudentUnKexiaoListParam param) {
        param.validate();
        //如果,统计一个校区的学员&&查询条件为空,则求该校区的待消汇总数据
        //如果,统计一个班级的学员&&查询条件为空,则求该班级的待消汇总数据
        //否则,根据查询条件查出学员,计算这些学员的待消汇总数据
        if (param.getScale().intValue() == StatisticsScaleEnum.ORG.getCode() && param.isEmptyConditionQuery()) {
            return this.getOrgSelfUnKexiaoInfo(param.getOrgId());
        }
        if (param.getScale().intValue() == StatisticsScaleEnum.CLASS.getCode() && param.isEmptyConditionQuery()) {
            return this.getClassUnKexiaoInfo(param.getClassId());
        }
        UnKexiaoStatisticsDto result = null;
        OrgAccount org = orgAccountDao.getById(param.getOrgId());

        if (org == null) {
            result = UnKexiaoStatisticsDto.createWithZeroValue();
        } else {
            List<Long> studentUserIds = Lists.newArrayList();
            if (param.getScale().intValue() == StatisticsScaleEnum.ORG.getCode()) {
                studentUserIds = orgStudentDao.listUserIdIgnoreStatus(param.getOrgId(), null, param.getKeyword(), null);
            } else if (param.getScale().intValue() == StatisticsScaleEnum.CLASS.getCode()) {
                //班级学员
                List<OrgStudentCourse> classStudent = orgStudentCourseDao.listByCourseId(param.getOrgId(), param.getClassId(), null, null);
                studentUserIds = classStudent.stream().map(OrgStudentCourse::getUserId).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(studentUserIds)) {
                    studentUserIds = orgStudentDao.listUserIdIgnoreStatus(param.getOrgId(), studentUserIds, param.getKeyword(), null);
                }
            }
            if (CollectionUtils.isEmpty(studentUserIds)) {
                result = UnKexiaoStatisticsDto.createWithZeroValue();
            } else {
                if (param.getScale().intValue() == StatisticsScaleEnum.ORG.getCode()) {
                    result = this.getUnKexiaoSumOfStudents(studentUserIds);
                } else {
                    result = this.getUnKexiaoSumOfClassStudents(param.getClassId(), studentUserIds);
                }
            }
        }
        return result;
    }

}
