/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.biz.consult.sync.impl;

import com.baijia.tianxiao.biz.consult.sync.SyncService;
import com.baijia.tianxiao.biz.consult.user.service.ConsultUserService;
import com.baijia.tianxiao.constants.TianXiaoConstant;
import com.baijia.tianxiao.dal.im.dao.ImMessageInfoDao;
import com.baijia.tianxiao.dal.im.po.ImMessageInfo;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.push.constant.MessageSource;
import com.baijia.tianxiao.dal.roster.dao.TxConsultUserDao;
import com.baijia.tianxiao.dal.roster.po.TxConsultUser;
import com.baijia.tianxiao.dal.sync.constant.MsgSyncType;
import com.baijia.tianxiao.dal.sync.dao.TxMsgSyncTimestampDao;
import com.baijia.tianxiao.dal.sync.po.TxMsgSyncTimestamp;
import com.baijia.tianxiao.dal.user.dao.StudentDao;
import com.baijia.tianxiao.dal.user.dao.UserDao;
import com.baijia.tianxiao.dal.user.po.Student;
import com.baijia.tianxiao.dal.user.po.User;
import com.baijia.tianxiao.util.collection.CollectorUtil;

import com.google.common.base.Function;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.TransformerUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * @author cxm
 * @version 1.0
 * @title SyncIMUserInfoServiceImpl
 * @desc TODO
 * @date 2015年12月8日
 */
@Service("syncIMUserInfoService")
@Slf4j
public class SyncIMUserInfoServiceImpl implements SyncService {

    @Resource
    private TxMsgSyncTimestampDao txMsgSyncTimestampDao;

    @Resource
    private ImMessageInfoDao imMessageInfoDao;

    @Resource
    private OrgAccountDao orgAccountDao;

    @Resource
    private UserDao userDao;

    @Resource
    private StudentDao studentDao;

    @Resource
    private OrgInfoDao orgInfoDao;

    @Resource
    private TxConsultUserDao consultUserDao;
    
    @Resource
    private OrgStudentDao orgStudentDao;
    
    @Resource
    private ConsultUserService consultUserService;

    private AtomicBoolean flag = new AtomicBoolean(false);

    @Override
    public void sync() {
        TxMsgSyncTimestamp syncInfo = getSyncInfo(MsgSyncType.IM_STU);
        if (!flag.get()) {
            log.info("sync im student info :{}", syncInfo);
            flag.set(true);
            try {
                syncTask(syncInfo);
            } catch (Exception e) {
                log.error("catch error:", e);
            } finally {
                flag.set(false);
            }
        } else {
            log.warn("has not finish,skip");
        }
        log.info("save sync info:{}", syncInfo);
        txMsgSyncTimestampDao.saveOrUpdate(syncInfo);
    }

    /**
     * @param syncInfo
     */
    private void syncTask(TxMsgSyncTimestamp syncInfo) {
        List<ImMessageInfo> messageInfos =
                imMessageInfoDao.getOrgStudentInfoListByMinId(syncInfo.getSyncId(), syncInfo.getSyncTime(), 100);
        if (CollectionUtils.isNotEmpty(messageInfos)) {
            Set<Long> studentNumbers = Sets.newHashSet();
            Set<Integer> orgNumbers = Sets.newHashSet();
            for (ImMessageInfo msgInfo : messageInfos) {
                if (msgInfo.getMsgId() > syncInfo.getSyncId()) {
                    syncInfo.setSyncId(msgInfo.getMsgId());
                }
                studentNumbers.add(msgInfo.getSender());
                orgNumbers.add(msgInfo.getReceiver().intValue());
            }
            List<User> users = userDao.getByNumbers(studentNumbers, "id", "number", "mobile");
            List<OrgAccount> accounts = orgAccountDao.getAccountByNumbers(orgNumbers, "id", "number");
            checkAndSaveConsulters(messageInfos, users, accounts);
        }
    }

    private void checkAndSaveConsulters(List<ImMessageInfo> messageInfos, List<User> users, List<OrgAccount> accounts) {

        Map<Long, User> userNumberIdMap = Maps.newHashMap();
        Set<Long> userIds = Sets.newHashSet();
        for (User u : users) {
            userIds.add(u.getId());
            userNumberIdMap.put(u.getNumber(), u);
        }

        List<Student> students = studentDao.getByUserIds(userIds, "userId", "realName", "nickName");

        Map<Long, Long> orgIdMap = CollectorUtil.collectMap(accounts, new Function<OrgAccount, Long>() {
            public Long apply(OrgAccount input) {
                return input.getNumber().longValue();
            }
        }, new Function<OrgAccount, Long>() {
            public Long apply(OrgAccount input) {
                return input.getId().longValue();
            }
        });
        Map<Long, Student> studentMap =
                CollectorUtil.collectMap(students,input -> input.getUserId());
        log.debug("student map:{}", studentMap);
        TxConsultUser consulter = null;
        int successCount = 0;
        for (ImMessageInfo info : messageInfos) {
            try {
                consulter = new TxConsultUser();
                User user = userNumberIdMap.get(info.getSender());

                /**
                 * 判重复
                 */
                List<TxConsultUser> txConsultUsers = consultUserDao.lookByUserId(orgIdMap.get(info.getReceiver()), user.getId());
                if (CollectionUtils.isNotEmpty(txConsultUsers)) {
                    continue;
                }

                Student stu = studentMap.get(user.getId());
                consulter.setConsultSource(MessageSource.ONLINE_IM.getValue());
                consulter.setMobile(user.getMobile());
                if (stu != null) {
                    consulter.setNickName(stu.getNickName());
                    consulter.setName(getNameFromStudent(stu));
                }
                consulter.setUserId(user.getId());
                consulter.setUserNumber(user.getNumber());
                consulter.setConsultStatus(0);
                consulter.setOrgId(orgIdMap.get(info.getReceiver()));
                OrgStudent orgStudent = this.orgStudentDao.getStudent(orgIdMap.get(info.getReceiver()), user.getId(), DeleteStatus.NORMAL.getValue());
                if (orgStudent != null) {
                    consulter.setStudentId(orgStudent.getId());
                    consulter.setNextRemindTime(orgStudent.getNextRemindTime());
                    if (orgStudent.getOrigin() != 0) {
                        orgStudent.setOrigin(0);
                        this.orgStudentDao.update(orgStudent, "origin");
                    }
                } 
                
                consultUserDao.save(consulter);
                if(consulter.getNextRemindTime() != null){
                    this.consultUserService.addSysBacklog(orgIdMap.get(info.getReceiver()), consulter.getId());
                }
                successCount++;
                log.info("save consulter:{} success", consulter);
            } catch (Exception e) {
                log.error("save consulter:{} catch error:{}", consulter, e);
            }
        }
        log.info("success save consulter size:{}", successCount);
    }
    
    protected String getNameFromStudent(Student student) {
        if (student != null) {
            if (StringUtils.isNoneBlank(student.getNickName())) {
                return student.getNickName();
            } else if (StringUtils.isNoneBlank(student.getRealName())) {
                return student.getRealName();
            }
        }
        return TianXiaoConstant.APPOINTMENT_STUDENT_NAME;
    }

    private TxMsgSyncTimestamp getSyncInfo(MsgSyncType syncType) {
        TxMsgSyncTimestamp syncInfo = txMsgSyncTimestampDao.getSyncTimestampByType(syncType.getSyncType());
        if (syncInfo == null) {
            syncInfo = new TxMsgSyncTimestamp();
            syncInfo.setSyncId(0l);
            syncInfo.setSyncTime(new Date());
            syncInfo.setSyncType(syncType.getSyncType());
        }
        return syncInfo;
    }

}
