/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.consult.user.service.impl;
import com.baijia.tianxiao.biz.consult.enums.SortOperation;
import com.baijia.tianxiao.biz.consult.user.service.ConsultCustomSourceService;
import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.dal.org.po.TXSaleClueRule;
import com.baijia.tianxiao.dal.push.constant.MessageSource;
import com.baijia.tianxiao.dal.roster.dao.ConsultCustomSourceDao;
import com.baijia.tianxiao.dal.roster.dao.TxConsultUserDao;
import com.baijia.tianxiao.dal.roster.po.ConsultCustomSource;
import com.baijia.tianxiao.exception.ParameterException;
import com.baijia.tianxiao.exception.PermissionException;
import com.baijia.tianxiao.sal.consult.dto.ConsultCustomSourceDto;
import com.baijia.tianxiao.sal.consult.service.ConsultSourceService;
import com.baijia.tianxiao.sal.organization.org.service.TXSaleClueRuleService;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

/**   
 * @title       : ConsultCustomSourceServiceImpl 
 * @description : 
 * @author      : zhenyujian
 * @date        : 2016年7月26日 上午12:09:22 
 */
@Slf4j
@Service
public class ConsultCustomSourceServiceImpl implements ConsultCustomSourceService {
	
	@Autowired
	private ConsultCustomSourceDao customSourceDao;
	@Autowired
	private TxConsultUserDao txConsultUserDao;
	
	@Autowired
	private TXSaleClueRuleService saleClueRuleService;
	@Autowired                                                                                                                                                                                                                                                                                                                                                                                                                         
	private ConsultSourceService consultSourceService;

	
	
	
	/*
	 *  线索来源 - 添加
	 */
	@Override
	@Transactional(rollbackFor = Exception.class)
	public ConsultCustomSourceDto add(Long orgId, String label) {
		Date now = new Date();

		//权限判断
		ConsultCustomSource.vaildateLabel(label);
		
		ConsultCustomSource source = customSourceDao.getBy(orgId, label);
		if(source!=null){
			throw new ParameterException("指定名称已被使用,请勿重复创建。");
		}
		//业务逻辑
		source = new ConsultCustomSource();
		source.setIsPaused(Flag.FALSE.getInt());
		source.setLabel(label);
		source.setOrgId(orgId);
		source.setCreateTime(now);
		source.setUpdateTime(now);
		
		customSourceDao.save(source);
		log.info("customSourceDao.save - source:{}",source);
		
		return ConsultCustomSourceDto.buildBy(source);
	}

	
	
	/*
	 *  线索来源 - 删除
	 */
	@Override
	@Transactional(rollbackFor = Exception.class)
	public void del(Long orgId, Long id) {
		if( txConsultUserDao.countBySource(orgId, id)>0 ){
			throw new PermissionException("该线索来源已被使用，无法删除，仅可停用。");
		}
		customSourceDao.delById(id);
	}

	
	
	/*
	 *  线索来源 - 修改
	 */
	@Override
	@Transactional(rollbackFor = Exception.class)
	public ConsultCustomSourceDto edit(Long orgId, Long id, String label) {
		Date now = new Date();

		//权限判断
		ConsultCustomSource.vaildateLabel(label);
		
		ConsultCustomSource source = customSourceDao.getById(id);
		if(source==null){
			throw new ParameterException("数据不存在");
		}
		
		ConsultCustomSource sourceByLabel = customSourceDao.getBy(orgId, label);
		if(sourceByLabel!=null){
			if(sourceByLabel.getId().equals(id)){
				return ConsultCustomSourceDto.buildBy(source);
			}else{
				throw new ParameterException("指定名称已被使用,请勿重复创建。");
			}
		}
		
		
		//业务逻辑
		source.setLabel(label);
		source.setUpdateTime(now);
		
		customSourceDao.update(source);
		log.info("customSourceDao.update - source:{}",source);
		
		return ConsultCustomSourceDto.buildBy(source);
	}

	
	
	/*
	 *  线索来源 - 停用、启用
	 */
	@Override
	@Transactional(rollbackFor = Exception.class)
	public ConsultCustomSourceDto pause(Long orgId, Long id, Integer isPaused) {
		Date now = new Date();

		//权限判断
		ConsultCustomSource source = customSourceDao.getById(id);
		if(source==null){
			throw new ParameterException("数据不存在");
		}
		if(isPaused==null || source.getIsPaused().equals(isPaused.intValue()) || !Flag.isBooleanInt(isPaused)){
			return ConsultCustomSourceDto.buildBy(source);
		}
		
		//业务逻辑
		source.setIsPaused(isPaused);
		source.setUpdateTime(now);
		customSourceDao.update(source);
		log.info("customSourceDao.pause - source:{}",source);
		
		//排序
		if(Flag.getBoolean(isPaused)){
			sortUpdate(orgId, id, SortOperation.DEL);
		}else{
			sortUpdate(orgId, id, SortOperation.ADD_RIGHT);
		}
		
		return ConsultCustomSourceDto.buildBy(source);
	}


	
	void sortUpdate(Long orgId, Long sourceId, SortOperation operation){
		TXSaleClueRule rule = saleClueRuleService.getByOrgId(orgId.intValue());
		String sort = StringUtils.isNotBlank(rule.getConsultSourceSort()) ? rule.getConsultSourceSort().trim() : "";
		
		//清除原排序串中 目标id
		String sortClone = String.format("%s%s%s", TXSaleClueRule.CONSULT_SOURCE_SORT_SPLIT, sort, TXSaleClueRule.CONSULT_SOURCE_SORT_SPLIT);
		String target = String.format("%s%s%s", TXSaleClueRule.CONSULT_SOURCE_SORT_SPLIT, sourceId, TXSaleClueRule.CONSULT_SOURCE_SORT_SPLIT);
		if(sortClone.equals(target)){
			sort = "";
		}else if(sortClone.contains(target)){
			sortClone.replace(target, TXSaleClueRule.CONSULT_SOURCE_SORT_SPLIT);
			sort = sortClone.substring(1,sortClone.length()-1);
		}
		
		//拼装sort
		switch (operation) {
			case DEL:
				
				break;
				
			case ADD_LEFT:
				sort = String.format("%s%s%s", TXSaleClueRule.CONSULT_SOURCE_SORT_SPLIT, sourceId, sort);
				break;
				
			case ADD_RIGHT:
				sort = String.format("%s%s%s", sort, TXSaleClueRule.CONSULT_SOURCE_SORT_SPLIT, sourceId);
				break;
	
			default:
				return;
		}
		rule.setConsultSourceSort(sort);
		saleClueRuleService.saveTXSaleClueRule(rule);
	}
	
	
	/*
	 *  线索来源 - 调整顺序
	 */
	@Override
	@Transactional(rollbackFor = Exception.class)
	public void sort(Long orgId, String sort) {
		Date now = new Date();
		if(StringUtils.isBlank(sort)){
			return;
		}
		
		//业务逻辑
		TXSaleClueRule rule = saleClueRuleService.getByOrgId(orgId.intValue());
		rule.setConsultSourceSort(sort);
		if(rule.getId()!=null){
			rule.setUpdateTime(now);
		}
		saleClueRuleService.saveTXSaleClueRule(rule);
	}

	
	
	/*
	 *  线索来源 - 列表
	 */
	@Override
	@Transactional(rollbackFor = Exception.class)
	public List<ConsultCustomSourceDto> selection(Long orgId, Integer scope, Integer isPaused) {
		List<ConsultCustomSourceDto> dtos = new ArrayList<ConsultCustomSourceDto>();
		
		if(isPaused!=null && !isPaused.equals(Flag.TRUE.getInt()) && !isPaused.equals(Flag.FALSE.getInt())){
			isPaused = null;
		}
		
		//查询 (scope 请看接口文档吧吧吧吧。。)
		if(scope!=null && scope == 0){
			dtos.addAll( consultSourceService.listConsultSystemSource(true) );
			
		}else if(scope!=null && scope == 1){
			List<ConsultCustomSource> list = customSourceDao.list(orgId, isPaused);
			for(ConsultCustomSource ccSource : list){
				dtos.add( ConsultCustomSourceDto.buildBy(ccSource) );
			}
			
		}else{
			List<ConsultCustomSource> list = customSourceDao.list(orgId, isPaused);
			for(ConsultCustomSource ccSource : list){
				dtos.add( ConsultCustomSourceDto.buildBy(ccSource) );
			}
			
			dtos.addAll( consultSourceService.listConsultSystemSource(true) );
		}
		
		//排序
		if( CollectionUtils.isNotEmpty(dtos) ){
			Map<Long,ConsultCustomSourceDto> dtosMap = new LinkedHashMap<Long, ConsultCustomSourceDto>();
			for(ConsultCustomSourceDto dto : dtos){
				dtosMap.put(dto.getId(), dto);
			}
			
			//设置sort
			TXSaleClueRule rule = saleClueRuleService.getByOrgId(orgId.intValue());
			String sort = rule.getConsultSourceSort();
			if(StringUtils.isNotBlank(sort)){
				int i=1;
				Long sourceId = null;
				ConsultCustomSourceDto dto = null;
				for( String s:sort.split(TXSaleClueRule.CONSULT_SOURCE_SORT_SPLIT) ){
					sourceId = Long.parseLong(s);
					dto = dtosMap.get(sourceId.longValue());
					if( dto!=null && !Flag.getBoolean(dto.getIsPaused()) ){
						dto.setSort(i);
						i++;
					}
				}
			}
			
			//排序
			sortConsultCustomSource(dtos);
		}
		
		return dtos;
	}
	
	//线索来源列表排序
	void sortConsultCustomSource(List<ConsultCustomSourceDto> dtos){
		
		for(ConsultCustomSourceDto dto:dtos){
			if(dto.getSort()==null){
				dto.setSort(Flag.NULL.getInt());
			}
		}
		
		Comparator<ConsultCustomSourceDto> comparator = new Comparator<ConsultCustomSourceDto>() {
            @Override
            public int compare(ConsultCustomSourceDto o1, ConsultCustomSourceDto o2) {
            	if( Flag.getBoolean(o1.getIsPaused()) && Flag.getBoolean(o2.getIsPaused()) ){//冻结vs冻结
            		if(o1.getUpdateTime().getTime() > o2.getUpdateTime().getTime()){
            			return 1;
            		}else{
            			return -1;
            		}
            		
            	}else if( Flag.getBoolean(o1.getIsPaused()) ){
            		return 1;
            		
            	}else if( Flag.getBoolean(o2.getIsPaused()) ){
            		return -1;
            		
            	}else{
            		if(o1.getSort()!=null && o2.getSort()!=null){
            			if(o1.getSort().equals(o2.getSort())){
            				return o1.getId().longValue() > o2.getId().longValue() ? -1:1;
            			}
            			return o1.getSort().intValue() > o2.getSort().intValue() ? 1:-1;
            		}else if( o1.getSort()!=null ){
	            		return 1;
	            		
	            	}else if( o2.getSort()!=null ){
	            		return -1;
	            		
	            	}else{
	            		return o1.getId().longValue() > o2.getId().longValue() ? -1:1;
	            	}
            		
            	}
            }

        };
        Collections.sort(dtos, comparator);
	}
	
	
	
	/*
	 *  线索来源 - 获取名称 
	 */
	@Transactional(readOnly=true)
	@Override
	public String getConsultSourceStr(Long id) {
		String str = null;
		MessageSource source = MessageSource.getByType(id.intValue());
		if(source!=null && source.isConsultSource()){
			str = source.getDesc();
		}else{
			ConsultCustomSource csource = customSourceDao.getById(id);
			if(csource!=null){
				str = csource.getLabel();
			}
		}
		return str;
	}
	
}
