/**
 * kuaikeguanjia.com Inc. Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.entity.chat;

import cn.kinyun.wework.sdk.entity.chat.msg.*;
import com.fasterxml.jackson.annotation.JsonAlias;
import cn.kinyun.wework.sdk.enums.ChatMsgType;
import cn.kinyun.wework.sdk.enums.ChatAction;
import cn.kinyun.wework.sdk.enums.TalkerType;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.util.List;
import java.util.stream.Collectors;

import lombok.Data;

/**
 * 解密后的聊天消息
 * 
 * @title ChatMsg
 * @author yanmaoyuan
 * @date 2021年3月18日
 * @version 1.0
 */
@Data
public class ChatMsg implements Serializable {

    private static final long serialVersionUID = 1L;

    public static final String EXTERNAL = "external";

    public static final String CONVERSATION_JOIN = "$$";
 
    /**
     * 消息id，消息的唯一标识，企业可以使用此字段进行消息去重。String类型
     */
    @JsonAlias("msgid")
    private String msgId;

    /**
     * 消息动作，目前有send(发送消息)/recall(撤回消息)/switch(切换企业日志)三种类型。String类型
     * 
     * @see ChatAction
     */
    private String action;

    /**
     * 消息发送方id。同一企业内容为userid，非相同企业为external_userid。消息如果是机器人发出，也为external_userid。String类型
     */
    private String from;


    /**
     * 消息接收方列表，可能是多个，同一个企业内容为userid，非相同企业为external_userid。数组，内容为string类型
     */
    private List<String> tolist;

    /**
     * 群聊消息的群id。如果是单聊则为空。String类型
     */
    @JsonAlias("roomid")
    private String roomId;

    /**
     * 消息发送时间戳，utc时间，ms单位。
     */
    @JsonAlias("msgtime")
    private Long msgTime;

    /**
     * 消息类型
     * 
     * @see ChatMsgType
     */
    @JsonAlias("msgtype")
    private String msgType;

    /**
     * 消息发送时间戳，utc时间，ms单位。
     * 仅在 action == "switch" 时有此字段
     */
    private Long time;

    private Long updateTime; //消息入库时间，可能拉取消息延迟较大等

    private Long seq;  //消息序号

    private String corpId; //企业Id

    private Long bizId;

    /**
     * 构造规则：群消息群Id,私聊消息对from和talkId排序后拼接
     *
     * 使用场景，统一群聊和单聊的情景，查询会话详情
     */
    private String conversationId; //会话Id

    private Boolean isRevoked;  //是否是撤回消息

    private Integer talkerType;  //群聊，单聊 1:联系人，2:群

    private Boolean externalContact;  //内部外部联系人，1：外部，0：内部

    private String fileUrl; //文件url
    private String fileName; //文件名 后面扩展使用
    private String fileType; //文件类型 txt pdf doc等
    private String transcodeFileUrl; //转码后url
    private Integer duration;

    /**
     * 具体为切换企业的成员的userid。String类型
     * 仅在 action == "switch" 时有此字段
     */
    private String user;

    private Text text;// 文本

    private Image image;// 图片

    private Revoke revoke;// 撤回消息

    private Agree agree;// 同意会话聊天内容

    private Disagree disagree;// 不同意会话聊天内容

    private Voice voice;// 语音

    private Video video;// 视频

    private Card card;// 名片
    
    private Location location;// 位置
    
    private Emotion emotion;// 表情

    private File file;// 文件

    private Link link;// 链接

    private Weapp weapp;// 小程序消息

    @JsonAlias("chatrecord")
    private ChatRecord chatRecord;// 会话记录消息

    private Todo todo;// 代办消息

    private Vote vote;// 投票消息

    private Collect collect;// 填表消息

    @JsonAlias("redpacket")
    private RedPacket redPacket;// 红包消息，互通红包消息

    private Meeting meeting;// 会议邀请消息

    private DocMsg doc;// 在线文档消息

    private Info info;// MarkDown消息, 图文消息，音视频通话消息。

    private Calendar calendar;// 日程消息

    private Mixed mixed;// 混合消息
    
    /**
     * String类型, 音频id
     */
    @JsonAlias("voiceid")
    private String voiceId;

    @JsonAlias("meeting_voice_call")
    private MeetingVoiceCall meetingVoiceCall;// 音频存档消息

    /**
     * String类型, 音频id。适用于voip_doc_share类型
     */
    @JsonAlias("voipid")
    private String voipId;

    @JsonAlias("voip_doc_share")
    private VoipDocShare voipDocShare;// 音频共享文档消息

    @JsonAlias("sphfeed")
    private SphFeed sphfeed;// 视频号消息

    private String encryptCorpId;

    private String encryptFormId;

    private String encryptToId;

    public static String getConversationId(String roomId, String fromId, String talkerId) {
        if (StringUtils.isBlank(roomId)) {
            return Lists.newArrayList(fromId, talkerId).stream().sorted().collect(Collectors.joining(CONVERSATION_JOIN));
        }
        return roomId;
    }
    public static String getConversationId(ChatMsg chatMsg) {
        if (!ChatAction.SEND.getValue().equals(chatMsg.getAction())) {
            return null;
        }
        if (StringUtils.isBlank(chatMsg.getRoomId()) && chatMsg.getTolist() != null && chatMsg.getTolist().size() > 0) {
            return Lists.newArrayList(chatMsg.getFrom(), chatMsg.getTolist().get(0)).stream().sorted().collect(Collectors.joining(CONVERSATION_JOIN));
        }
        return chatMsg.getRoomId();
    }
    public static int getTalkerTypeByConversationId(String conversationId) {
        if (conversationId.startsWith("wr") && conversationId.indexOf(CONVERSATION_JOIN) == -1) {
            return TalkerType.CHATROOM.getValue();
        }
        return TalkerType.CONTACT.getValue();
    }
}
