package cn.kinyun.pay.business.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

import java.math.BigInteger;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPrivateKeySpec;
import java.security.spec.RSAPublicKeySpec;

/**
 * 数字签名工具类
 */
@SuppressWarnings({"java:S4790"})
@Slf4j
public class DigitalSignatureUtil {
    private final static String KEY_ALGORITHM = "RSA";
    private final static String SIGNATURE_ALGORITHM = "SHA1withRSA";
    private final static Integer RSA_SIZE = 1024;

    private DigitalSignatureUtil() {

    }

    //生成密钥对
    public static KeyPair getKeyPair() throws Exception {
        KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA");
        keyGen.initialize(RSA_SIZE); //可以理解为：加密后的密文长度，实际原文要小些 越大 加密解密越慢
        KeyPair keyPair = keyGen.generateKeyPair();
        return keyPair;
    }

    /**
     * 获取新的私钥、公钥，左私钥，右公钥
     *
     * @return Pair<String, String> <私钥,公钥>
     * @throws Exception
     */
    public static Pair<String, String> getTupleKey() throws Exception {
        KeyPair keyPair = getKeyPair();
        String privateKey = Base64.encodeBase64String(keyPair.getPrivate().getEncoded());
        String publicKey = Base64.encodeBase64String(keyPair.getPublic().getEncoded());
        return new ImmutablePair<String, String>(privateKey, publicKey);
    }

    /**
     * 请求报文签名
     *
     * @param obj
     * @param privateKey
     * @return
     * @throws Exception
     */
    public static String signFromSha1WithRSA(Object obj, String privateKey) {
        String sign;
        try {
            sign = getSha1WithRSASign(JsonSortUtil.sortBy(obj), privateKey);
        } catch (Exception e) {
            throw new RuntimeException("签名异常", e);
        }
        return sign;
    }

    /**
     * 使用SHA1WithRSA算法生成数字签名
     *
     * @param content    待加密字符串
     * @param privateKey 私钥通过BASE64编码成功的字符串
     * @return 数字签名
     * @throws Exception
     */
    public static String getSha1WithRSASign(String content, String privateKey) throws Exception {
        byte[] privateKeyBytes = Base64.decodeBase64(privateKey);
        byte[] contentBytes = content.getBytes("utf-8");
        PKCS8EncodedKeySpec pkcs8EncodedKeySpec = new PKCS8EncodedKeySpec(privateKeyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
        signature.initSign(keyFactory.generatePrivate(pkcs8EncodedKeySpec));
        signature.update(contentBytes);
        byte[] signs = signature.sign();
        return Base64.encodeBase64String(signs);
    }

    /**
     * 由私钥计算公钥
     *
     * @param privateKey
     * @return
     * @throws Exception
     */
    public static String calPublicKeyFromPrivateKey(String privateKey) throws Exception {
        byte[] privateKeyBytes = Base64.decodeBase64(privateKey);
        PKCS8EncodedKeySpec pkcs8EncodedKeySpec = new PKCS8EncodedKeySpec(privateKeyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        PrivateKey privateKey1 = keyFactory.generatePrivate(pkcs8EncodedKeySpec);
        KeyFactory kf = KeyFactory.getInstance(KEY_ALGORITHM);
        RSAPrivateKeySpec priv = kf.getKeySpec(privateKey1, RSAPrivateKeySpec.class);
        RSAPublicKeySpec keySpec = new RSAPublicKeySpec(priv.getModulus(), BigInteger.valueOf(65537));
        PublicKey publicKey = kf.generatePublic(keySpec);
        return Base64.encodeBase64String(publicKey.getEncoded());
    }

}
